<?php
namespace boru\query;

use boru\dhutils\traits\JsonTrait;
use boru\query\models\Value;
use boru\query\Record;
use boru\query\Query;
use boru\dot\Dot;
use boru\query\models\EntityDefinition;

class Entity implements \JsonSerializable {
    use JsonTrait;
    /** @var Record */
    private $record;
    /** @var bool */
    private $isNew = true;

    private $dataBeforeSave = [];
    private $dataSaveDelta = [];

    private $entityDefinition;
    private $entityIsInitialized = false;

    private $handlerStack = [
        "load" => [],
        "create" => [],
        "update" => [],
        "delete" => [],
        "beforeSave" => [],
        "afterSave" => []
    ];


    protected static $thisEntityDefinition = null;
    /**
     * This method should return an array of entity definition options or an EntityDefinition object
     * @return EntityDefinition 
     */
    public static function entityDefinition() {
        //find the caller and echo it
        return new EntityDefinition(["className" => static::class]);
    }

    public function __construct($data=[],$definitionOrQuery=null) {
        $this->initEntity($definitionOrQuery);
        $query = $this->entityQuery();
        if($data instanceof Record) {
            $this->record = $data;
        } elseif(is_array($data) || empty($data)) {
            $this->record = new Record($query,$data);
        } elseif(is_numeric($data)) {
            $this->record = new Record($query);
            $this->record->load($data);
        } elseif($this->property("labelField")) {
            $this->record = $query->where($this->property("labelField"),$data)->toRecord();
        } else {
            throw new \Exception("Invalid data type for EntityModule constructor");
        }
        if($this->record->_id()) {
            $this->isNew = false;
        }
        $this->triggerEvent("load");
    }

    public function fromArray($data=[],$saveOnSet=false) {
       $this->multiSet($data,$saveOnSet);
    }

    protected function getEntityDefinition() {
        if($this->entityDefinition === null) {
            return $this->entityDefinition();
        }
        return $this->entityDefinition;
    }
    protected static function getStaticEntityDefinition() {
        if(static::$thisEntityDefinition === null) {
            static::$thisEntityDefinition = static::entityDefinition();
        }
        return static::$thisEntityDefinition;
    }
    public function initEntity($definitionOrQuery=null) {
        if($this->entityIsInitialized) {
            return;
        }
        if($definitionOrQuery === null) {
            $definitionOrQuery = static::getStaticEntityDefinition();
        }
        if($definitionOrQuery instanceof Query) {
            $this->entityDefinition = Factory::registerQuery($definitionOrQuery);
        } elseif($definitionOrQuery !== null) {
            $this->entityDefinition = Factory::register($definitionOrQuery);
        } else {
            $this->entityDefinition = Factory::register($this);
        }
        $this->entityIsInitialized = true;
    }
    
    public function property($key=null,$default=null) {
        $this->initEntity();
        return $this->entityDefinition->get($key,$default);
    }

    public function entityQuery() {
        return $this->entityDefinition->query();
    }

    public function id() {
        return $this->record->_id();
    }

    public function isNew() {
        return $this->isNew ? true : false;
    }
    public function setNew($isNew) {
        $this->isNew = $isNew ? true : false;
    }
    public function isChanged() {
        return count($this->dataSaveDelta) > 0 ? true : false;
    }
    public function changeDelta() {
        return $this->dataSaveDelta;
    }

    public function save() {
        $this->dataBeforeSave = $this->record->get();
        $this->dataSaveDelta = $this->calculateDelta();
        if(!$this->isNew && count($this->dataSaveDelta) == 0) {
            return $this;
        }
        $this->triggerEvent("beforeSave");
        if($this->isNew && $this->property("createdField")) {
            $this->record->set($this->property("createdField"),Value::functionValue("NOW()"));
        }
        if($this->property("updatedField")) {
            $this->record->set($this->property("updatedField"),Value::functionValue("NOW()"));
        }
        $this->record->save();
        if($this->id()) {
            $this->record->load($this->record->_id());
            if($this->isNew) {
                $this->triggerEvent("create");
            	$this->isNew = false;
            } else {
                $this->triggerEvent("update");
            }
        }
        $this->triggerEvent("afterSave");
        return $this;
    }

    public function get($key=null,$default=null) {
        if(strpos($key,".") !== false && $this->record) {
            $arr = explode(".",$key);
            if($this->record->get($arr[0]) !== null) {
                $key = $arr[0];
                $result = $this->transformFieldFromDb($this->record->get($key,$default),$key);
                if(is_array($result)) {
                    array_shift($arr);
                    $string = implode(".",$arr);
                    return Dot::get($result,$string,$default);
                }
                return $result;
            }
        }
        return $this->transformFieldFromDb($this->record->get($key,$default),$key);
    }

    public function set($key,$value,$save=null) {
        if($this->record) {
            if(strpos($key,".") !== false) {
                $keyArr = explode(".",$key,2);
                $result = $this->get($keyArr[0]);
                if(is_array($result)) {
                    $key = $keyArr[0];
                    $keyRest = $keyArr[1];
                    Dot::set($result,$keyRest,$value);
                    $value = $result;
                }
            }
            $this->record->set($key,$this->transformFieldToDb($value,$key));
            if($save !== false) {
                if($save === true || $this->property("saveOnSet") === true) {
                    $this->save();
                }
            }
            //$this->record->set($key,$value);
        }
        return $this;
    }

    public function multiSet($data,$saveOnSet=null) {
        if($saveOnSet === null) {
            $saveOnSet = static::property("saveOnSet");
        }
        foreach($data as $key => $value) {
            $this->set($key,$value,false);
        }
        if($saveOnSet) {
            $this->save();
        }
        return $this;
    }

    public function toArray() {
        return $this->record->get();
    }

    public function on($event,$callback) {
        if(!isset($this->handlerStack[$event])) {
            throw new \Exception("Invalid event type");
        }
        $this->handlerStack[$event][] = $callback;
        return $this;
    }

    public function triggerEvent($event) {
        $stack = $this->getEntityDefinition()->on($event);
        foreach($stack as $callback) {
            $callback($this);
        }
        foreach($this->handlerStack[$event] as $callback) {
            $callback($this);
        }
        return $this;
    }

    public function _saveOnSet($saveOnSet=null) {
        if($saveOnSet !== null) {
            $this->entityDefinition->set("saveOnSet",$saveOnSet);
        }
        return $this->entityDefinition->get("saveOnSet");
    }

    public function _isJsonField($field) {
        return in_array($field,$this->property("jsonFields",[]));
    }

    private function transformFieldFromDb($value,$field=null) {
        if(is_array($value)) {
            $result = [];
            foreach($value as $f) {
                $result[$f] = $this->transformFieldFromDb($f,$value);
            }
            return $result;
        }
        if($field === null) {
            return $value;
        }
        if($this->_isJsonField($field)) {
            return json_decode($value,true);
        }
        return $value;
    }

    private function transformFieldToDb($value,$field=null) {
        if($field === null) {
            return $value;
        }
        if($this->_isJsonField($field)) {
            return json_encode($value);
        }
        return $value;
    }

    private function calculateDelta() {
        $deltaChanged = [];
        //check for new fields
        foreach($this->record->get() as $key => $value) {
            if(!array_key_exists($key,$this->dataBeforeSave)) {
                $deltaChanged[$key] = $value;
            } elseif($this->dataBeforeSave[$key] != $value) {
                $deltaChanged[$key] = $value;
            }
        }
        return $deltaChanged;
    }

    public static function fromDefinition($definition) {
        $className = $definition["className"];
        return new $className(null,$definition);
    }
    public static function fromQuery($query) {
        $className = $query->getTables()->primaryTable()->name();
        return new $className(null,$query);
    }
}