<?php
namespace boru\query\models;

use boru\dhutils\traits\IteratorTrait;
use boru\dhutils\traits\JsonTrait;
use boru\query\Query;
use boru\query\Entity;

class EntityDefinition {
    use JsonTrait;
    use IteratorTrait;

    protected $data = [
        "name" => "",
        "className" => Entity::class,
        "queryBase" => null,
        "idField" => "id",
        "labelField" => "",
        "createdField" => "",
        "updatedField" => "",
        "jsonFields" => [],
        "saveOnSet" => true
    ];

    private $handlerStack = [
        "load" => [],
        "create" => [],
        "update" => [],
        "delete" => [],
        "beforeSave" => [],
        "afterSave" => []
    ];

    public function __construct($data=[]) {
        $this->fromArray($data);
        if(!$this->get("name")) {
            if($this->get("className")) {
                $reflection = new \ReflectionClass($this->get("className"));
                $this->set("name",$reflection->getShortName());
            } elseif($this->get("queryBase")) {
                $this->set("name",$this->get("queryBase")->getTables()->primaryTable()->name());
            }
        }
        if(!$this->get("name")) {
            throw new \Exception("Entity name not set");
        }
    }

    public function fromArray($array) {
        foreach($array as $key => $value) {
            $this->set($key,$value);
        }
        return $this;
    }

    public function toArray() {
        return $this->data;
    }

    public function name() {
        return $this->get("name");
    }

    /**
     * This method returns a clone of the queryBase property
     * @return Query|null 
     */
    public function queryBase() {
        if($this->get("queryBase") === null) {
            $query = new Query();
            $query->select("*");
            $query->from($this->name());
            return $query;
        }
        return clone $this->get("queryBase");
    }
    public function idField() {
        return $this->get("idField");
    }
    public function labelField() {
        return $this->get("labelField");
    }
    public function createdField() {
        return $this->get("createdField");
    }
    public function updatedField() {
        return $this->get("updatedField");
    }
    public function jsonFields() {
        return $this->get("jsonFields");
    }
    public function className() {
        return $this->get("className");
    }
    public function saveOnSet() {
        return $this->get("saveOnSet");
    }

    public function on($event,$callback=null) {
        if($callback === null) {
            if(!isset($this->handlerStack[$event])) {
                throw new \Exception("Invalid event");
            }
            return $this->handlerStack[$event];
        }
        if(!isset($this->handlerStack[$event])) {
            throw new \Exception("Invalid event");
        }
        $this->handlerStack[$event][] = $callback;
        return $this;
    }

    public function set($key,$value) {
        if(array_key_exists($key, $this->data)) {
            $this->data[$key] = $value;
        }
        return $this;
    }
    public function get($key=null,$default=null) {
        if($key === null) {
            return $this->data;
        }
        if(array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }
        return $default;
    }

    private static $classNameToDefinition = [];
    public static function fromEntity($entityObject) {
        $entityClass = get_class($entityObject);
        if(!isset(self::$classNameToDefinition[$entityClass])) {
            $entityDefinition = $entityObject->entityDefinition();
            if(is_array($entityDefinition)) {
                //set the class name
                $entityDefinition["className"] = $entityClass;

                //set the name
                if(!isset($entityDefinition["name"]) || empty($entityDefinition["name"])) {
                    $reflection = new \ReflectionClass($entityClass);
                    $entityDefinition["name"] = $reflection->getShortName();
                }
            } elseif($entityDefinition instanceof EntityDefinition) {

                //set the class name
                $entityDefinition->set("className",$entityClass);

                //set the name
                if(empty($entityDefinition->name())) {
                    $reflection = new \ReflectionClass($entityClass);
                    $entityDefinition->set("name",$reflection->getShortName());
                }
            } else {
                throw new \Exception("Invalid entity definition");
            }
            self::$classNameToDefinition[$entityClass] = new EntityDefinition($entityDefinition);
        }
        return self::$classNameToDefinition[$entityClass];
    }

    /**
     * This method is used to create an EntityDefinition object from a query
     * @param string $entityName 
     * @param Query $query 
     * @return EntityDefinition 
     */
    public static function fromQuery($entityName,$query) {
        if($query->getTables()->idColumn()) {
            $idField = $query->getTables()->idColumn();
        } else {
            $idField = "id";
        }
        $entityDefinition = new EntityDefinition([
            "name" => $entityName,
            "idField" => $idField,
            "queryBase" => $query
        ]);
        return $entityDefinition;
    }

    public function query() {
        $query = $this->queryBase();
        if($query->mode() != Query::MODE_SELECT) {
            if($query->getColumns()->count() == 0) {
                $query->select("*");
            }
            $query->mode(Query::MODE_SELECT);
        }
        $query->mode(Query::MODE_SELECT);
        $query->entityDefinition($this);
        return $query;
    }
}