<?php
namespace boru\query;

use boru\dhdb\dhDB;
use boru\dhutils\traits\JsonTrait;
use boru\query\models\Column;
use boru\query\models\Columns;
use boru\query\models\Condition;
use boru\query\models\Conditions;
use boru\query\models\Tables;
use boru\query\Query;
use boru\dot\Dot;
use boru\query\models\Value;
use Exception;

class Record implements \JsonSerializable {
    use JsonTrait;
    /** @var Columns */
    private $columns;
    /** @var Tables */
    private $tables;
    /** @var Conditions */
    private $where;
    /** @var array */
    private $rawData=[];
    /** @var Query */
    private $query;

    private $columnMap = [];
    private $columnRefToColumn = [];
    private $allTableColumns = [];
    private $tableRefToTable =[];

    private $insertMap = [];

    protected $data = [];

    /*
    Here is the plan.. The plan is for a Record to be able to be a "row" result from a Query.
    It will be able to have changes made to it, and then be able to be saved back to the database.
    The query will define the columns and tables that the record is based on.
    Additional metaInformation can be added such as createdTime and updatedTime columns.
    */

    public function __construct($options=[],$data=[]) {
        $this->columns = new Columns();
        $this->tables = new Tables();
        $this->where = new Conditions();
        $this->rawData = $data;
        if($options instanceof Query) {
            $this->_query($options);
        } elseif(is_array($options)) {
            if(isset($options["data"])) {
                $this->rawData = $options["data"];
            }
            if(isset($options["query"])) {
                $this->_query($options["query"]);
            }
            if(isset($options["tables"])) {
                $this->_tables($options["tables"]);
            }
            if(isset($options["where"])) {
                $this->_where($options["where"]);
            }
            if(isset($options["columns"])) {
                $this->_columns($options["columns"]); //triggers populate
            }
        }
    }

    public function toArray() {
        return $this->data;
    }

    public function _id() {
        //get the id column
        $idColumn = $this->tables->idColumn();
        return $this->get($idColumn);
    }

    /**
     * Load a record from the database.
     * @param mixed $value Can be an array of conditions, a single condition, or a value to match against the id column.
     * @param string $operator Default is "="
     * @param mixed $columnName Default is null. If a single condition is being used, this is the column name.
     * @return $this 
     * @throws Exception 
     */
    public function load($value, $operator="=", $columnName=null) {
        $this->where = new Conditions();
        $this->where->setQuery($this->query);
        if($columnName === null && ($column = $this->tables->idColumn()) !== null) {
            $this->where->addCondition([$column,$operator,$value]);
        } elseif($columnName instanceof Condition || $columnName instanceof Conditions) {
            $this->where->addCondition($columnName);
        } elseif($value instanceof Condition || $value instanceof Conditions) {
            $this->where->addCondition($value);
        } else {
            $this->where->addCondition([$columnName,$operator,$value]);
        }
        $result = $this->query->resetWhere()->where($this->where)->limit(1)->toResult();
        while($row = $result->next()) {
            $this->_populate($row->asArray());
            return $this;
        }
        return $this;
    }

    public function count() {
        $query = clone $this->query;
        if($this->query->getWhere()->count()<=0) {
            if($this->_id()) {
                $idColumn = $this->tables->idColumn();
                $query->where([$idColumn,"=",$this->_id()]);
            } else {
                throw new Exception("Cannot count without a where condition");
            }
        }
        return $query->toCount();
    }

    public function save() {
        if($this->_id()) {
            return $this->_update();
        }
        return $this->_insert();
    }

    private function _insert() {
        $tableInserts = $this->_makeInserts();
        $insertId = null;
        $firstTable = true;
        foreach($tableInserts as $tableRef=>$conditions) {
            if(!$firstTable) {
                foreach($conditions->conditions() as $condition) {
                    if($condition instanceof Condition && $condition->value() !== null) {
                        if($condition->value() instanceof Value && $condition->column() instanceof Column) {
                            if(isset($this->insertMap[$tableRef][$condition->column()->name()])) {
                                $value = $condition->value()->value();
                                $colDest = $this->insertMap[$tableRef][$condition->column()->refName()];
                                $newValue = $this->get($colDest);
                                $condition->value($newValue);
                            }
                        }
                    }
                }
            }
            if($firstTable) {
                $firstTable = false;
            }
            $query = Query::create();
            $table = $this->tableRefToTable[$tableRef];
            $query->insert($table->name())->set($conditions);
            //continue;
            $res = $query->run("INSERT");
            if($this->_id() === null) {
                $insertId = dhDB::instance()->lastInsertId();
                $idColumn = $this->tables->idColumn();
                $this->set($idColumn,$insertId);
            }
            
        }
    }

    private function _update() {
        $conditions = $this->_makeUpdate();
        $query = Query::create();
        $query->update($this->tables)->set($conditions);
        if($this->count()>1) {
            throw new Exception("Cannot update multiple records at once with a single record object");
        }
        $query->where($this->where);
        $res = $query->run("UPDATE");
        return $res;
    }

    public function _populate($data=[]) {
        if($data === null) {
            $data = [];
        }
        $this->rawData = $data;
        $this->tableRefToTable = [];
        $this->columnMap = [];
        $this->columnRefToColumn = [];
        $this->allTableColumns = [];
        $this->data = [];
        $tables = $this->tables->getAllTables();
        foreach($tables as $table) {
            $tableRef = $table->refName();
            $this->tableRefToTable[$tableRef] = $table;
        }
        $cols = $this->columns->getColumns();
        foreach($cols as $col) {
            if($col instanceof Column) {
                $value = null;
                $colRef = $col->refName();
                $tableRef = $col->tableRef();
                $this->columnMap[$colRef] = $tableRef;
                $this->columnRefToColumn[$colRef] = $col;
                if(array_key_exists($colRef,$this->rawData)) {
                    $value = $this->rawData[$colRef];
                } elseif($col->default()) {
                    $value = $col->default();
                }
                $this->set($colRef,$value);
            }
        }
        $tables = $this->tables->getAllTables();
        foreach($tables as $table) {
            $tableRef = $table->refName();
            $columns = $table->columns();
            foreach($columns as $column) {
                $colRef = $column->name();
                $this->allTableColumns[$tableRef][$colRef] = $column;
            }
        }
    }

    /**
     * Get the query object
     * @param Query|null $query 
     * @return Query 
     */
    public function _query($query=null) {
        if($query!==null) {
            if($query instanceof Query) {
                $this->query = $query;
                $this->_tables($query->getTables());
                $this->_columns($query->getColumns()); //triggers populate
            }
        }
        return $this->query;
    }

    /**
     * Get the columns object
     * @param Columns|null $columns 
     * @return Columns 
     */
    public function _columns($columns=null) {
        if($columns!==null) {
            if(!$this->tables) {
                throw new \Exception("Cannot set columns without tables. Please set tables first, or use the constructor to set tables and columns. Alternatively, you can set the query object using the constructor or the _query method.");
            }
            if(is_array($columns)) {
                $this->columns = new Columns();
                foreach($columns as $column) {
                    $this->columns->addColumn($column);
                }
            } elseif($columns instanceof Columns) {
                $this->columns = $columns;
            } else {
                throw new \Exception("Columns must be an array or an instance of Columns");
            }
            $this->columns = $columns;
            $this->_populate($this->rawData);
        }
        return $this->columns;
    }

    /**
     * Get the tables object
     * @param Tables|null $tables 
     * @return Tables 
     */
    public function _tables($tables=null) {
        if($tables!==null) {
            if(is_array($tables)) {
                $this->tables = new Tables();
                foreach($tables as $table) {
                    $this->tables->addTable($table);
                }
            } elseif($tables instanceof Tables) {
                $this->tables = $tables;
            } else {
                throw new \Exception("Tables must be an array or an instance of Tables");
            }
            $this->tables = $tables;
        }
        return $this->tables;
    }

    /**
     * Get the where object
     * @param Conditions|null $recordData 
     * @return Conditions 
     */
    public function _where($recordData=null) {
        if($recordData!==null) {
            $this->where = $recordData;
        }
        return $this->where;
    }

    /**
     * Get the schema of the record
     * @return array 
     */
    public function _schema() {
        $schema = [];
        foreach($this->columnMap as $colRef=>$tableRef) {
            $schema[$tableRef][$colRef] = $this->get($colRef);
        }
        return $schema;
    }

    /**
     * Make a set of conditions for updating the record
     * @return Conditions 
     */
    private function _makeUpdate() {
        $conditions = new Conditions();
        $conditions->setQuery($this->query);
        foreach($this->columnMap as $colRef=>$tableRef) {
            if($this->columnRefToColumn[$colRef]->isPrimary()) {
                continue;
            }
            $conditions->addCondition([$colRef,"=",$this->get($colRef)]);
        }
        return $conditions;
    }

    /**
     * Make a set of conditions for inserting the record
     * @return Conditions 
     */
    private function _makeInserts() {
        $tableConditions = [];
        $schema = $this->_schema();
        $joins = $this->tables->joins();
        $tables = $this->tables->tables();
        foreach($joins as $join) {
            foreach($join->on()->conditions() as $on) {
                if($on instanceof Condition) {
                    if($on->leftSide() instanceof Column && $on->leftSide()->table()->refName() != $join->table()->refName()) {
                        $tableRef = $on->rightSide()->table()->name();
                        $colRef = $on->rightSide()->name();
                        $sourceRef = $on->leftSide()->table()->name();
                        $schema[$tableRef][$colRef] = $this->get($sourceRef);
                        if(empty($schema[$tableRef][$colRef])) {
                            $schema[$tableRef][$colRef] = "{{".$on->leftSide()->name()."}}";
                        }
                    } elseif($on->rightSide() instanceof Column && $on->rightSide()->table()->refName() != $join->table()->refName()) {
                        $tableRef = $on->leftSide()->table()->name();
                        $colRef = $on->leftSide()->name();
                        $sourceRef = $on->rightSide()->table()->name();
                        $schema[$tableRef][$colRef] = $this->get($sourceRef);
                        if(empty($schema[$tableRef][$colRef])) {
                            $schema[$tableRef][$colRef] = "{{".$on->rightSide()->name()."}}";
                        }
                    }
                }
            }
        }
        foreach($schema as $tableRef=>$data) {
            $conditions = new Conditions();
            $conditions->setQuery($this->query);
            foreach($data as $colRef=>$value) {
                $column = isset($this->columnRefToColumn[$colRef]) ? $this->columnRefToColumn[$colRef] : null;
                if($column && $column->isPrimary() && substr($value,0,2) != "{{" && substr($value,-2) != "}}") {
                    continue;
                }
                if($value == "{{id}}") {
                    $this->insertMap[$tableRef][$colRef] = $this->tables->idColumn();
                }
                if($value === null) {
                    $value = Value::nullValue();
                }
                $conditions->addCondition([$colRef,"=",$value]);
            }
            $tableConditions[$tableRef] = $conditions;
        }
        return $tableConditions;
    }


    //Container functions

    /**
     * Get a value from the container using dot notation
     * @param mixed $key 
     * @param mixed $default 
     * @return mixed 
     */
    public function dotGet($key,$default=null) {
        return Dot::get($this->data,$key,$default);
    }

    /**
     * Set a value in the container using dot notation
     * @param mixed $key 
     * @param mixed $value 
     * @return array 
     */
    public function dotSet($key,$value) {
        Dot::set($this->data,$key,$value);
        return $this->data;
    }

    /**
     * Get a value from the container
     * @param mixed $key 
     * @return array 
     */
    public function get($name=null,$default=null) {
        if($name === null) {
            return $this->data;
        }
        if(array_key_exists($name, $this->data)) {
            return $this->data[$name];
        }
        return $this->dotGet($name,$default);
    }

    /**
     * Set a value in the container
     * @param mixed $name 
     * @param mixed $value 
     * @return self|array
     */
    public function set($name,$value,$inital=false) {
        if($inital) {
            $this->data[$name] = $value;
            return $this;
        }
        if(array_key_exists($name, $this->data)) {
            $this->data[$name] = $value;
            return $this;
        }
        return $this->dotSet($name,$value);
    }

    /**
     * Set a raw (unquoted) value in the container
     * @param mixed $name 
     * @param mixed $value 
     * @return self|array
     */
    public function setRaw($name,$value,$inital=false) {
        return $this->setFunction($name,$value,$inital);
    }

    /**
     * Set a function value in the container
     * @param mixed $name 
     * @param mixed $value 
     * @return self|array
     */
    public function setFunction($name,$value,$inital=false) {
        if(!$value instanceof Value) {
            $value = Value::functionValue($value);
        }
        if($inital) {
            $this->data[$name] = $value;
            return $this;
        }
        if(array_key_exists($name, $this->data)) {
            $this->data[$name] = $value;
            return $this;
        }
        return $this->dotSet($name,$value);
    }

    /**
     * Set a subquery value in the container
     * @param mixed $name 
     * @param mixed $value 
     * @return self|array
     */
    public function setSubquery($name,$value,$inital=false) {
        if(!$value instanceof Value) {
            $value = Value::subqueryValue($value);
        }
        if($inital) {
            $this->data[$name] = $value;
            return $this;
        }
        if(array_key_exists($name, $this->data)) {
            $this->data[$name] = $value;
            return $this;
        }
        return $this->dotSet($name,$value);
    }

    /**
     * Set a column value in the container
     * @param mixed $name 
     * @param mixed $value 
     * @return self|array
     */
    public function setColumn($name,$value,$inital=false) {
        if(!$value instanceof Value) {
            $value = Value::columnValue($value);
        }
        if($inital) {
            $this->data[$name] = $value;
            return $this;
        }
        if(array_key_exists($name, $this->data)) {
            $this->data[$name] = $value;
            return $this;
        }
        return $this->dotSet($name,$value);
    }
}