<?php
namespace boru\query\models;

class Operator {
    public static $comparatorToSql = [
        "eq" => "=",
        "ne" => "<>",
        "lt" => "<",
        "gt" => ">",
        "lte" => "<=",
        "gte" => ">=",
        "like" => "LIKE",
        "nlike" => "NOT LIKE",
        "bw" => "LIKE",
        "nbw" => "NOT LIKE",
        "ew" => "LIKE",
        "new" => "NOT LIKE",
        "in" => "IN",
        "nin" => "NOT IN",
        "is" => "IS",
        "nis" => "IS NOT",
        "between" => "BETWEEN"
    ];
    public static $prefixSuffix = [
        "eq" => ["",""],
        "ne" => ["",""],
        "lt" => ["",""],
        "gt" => ["",""],
        "lte" => ["",""],
        "gte" => ["",""],
        "like" => ["",""],
        "nlike" => ["",""],
        "bw" => ["","%"],
        "nbw" => ["","%"],
        "ew" => ["%",""],
        "new" => ["%",""],
        "in" => ["(",")"],
        "nin" => ["(",")"],
        "is" => ["",""],
        "nis" => ["",""],
        "between" => ["",""]
    ];
    public static $comparatorAliases = [
        "=" => "eq",
        "!=" => "ne",
        "<>" => "ne",
        "<" => "lt",
        ">" => "gt",
        "<=" => "lte",
        ">=" => "gte",
        "!eq" => "ne",
        "!like" => "nlike",
        "!in" => "nin",
        "!bw" => "nbw",
        "!ew" => "new",
        "like" => "like",
        "not like" => "nlike",
        "in" => "in",
        "not in" => "nin",
        "begins with" => "bw",
        "not begins with" => "nbw",
        "equals" => "eq",
        "neq" => "ne",
        "not equals" => "ne",
        "less than" => "lt",
        "greater than" => "gt",
        "less than or equals" => "lte",
        "greater than or equals" => "gte",
        "starts with" => "bw",
        "not starts with" => "nbw",
        "not begins with" => "nbw",
        "ends with" => "ew",
        "not ends with" => "new",
        "not begins with" => "new",
        "not in" => "nin",
        "not like" => "nlike",
        "is not" => "nis",
        "between" => "between",
    ];

    private $sqlComparator;
    private $comparator;

    public function __construct($comparator) {
        $this->comparator($comparator);
    }

    public function comparator($comparator=null) {
        if($comparator!==null) {
            if(array_key_exists($comparator,self::$comparatorAliases)) {
                $comparator = self::$comparatorAliases[$comparator];
            }
            if(!array_key_exists($comparator,self::$comparatorToSql)) {
                if(in_array($comparator,self::$comparatorToSql)) {
                    $comparator = array_search($comparator,self::$comparatorToSql);
                } else {
                    throw new \Exception("Unknown comparator: ".$comparator);
                }
            }
            $this->comparator = $comparator;
            $this->sqlComparator = self::$comparatorToSql[$comparator];
        }
        return $this->comparator;
    }

    public function toSql() {
        return $this->sqlComparator;
    }

    /**
     * 
     * @param mixed $value 
     * @return string 
     */
    public function adjustedValue($value,&$values=[]) {
        $prefix = $suffix = "";
        if(isset(self::$prefixSuffix[$this->comparator])) {
            $prefix = self::$prefixSuffix[$this->comparator][0];
            $suffix = self::$prefixSuffix[$this->comparator][1];
        }
        if($value instanceof Value) {
            if($value->needsPreparing()) {
                $valueString = $value->toSql($values);
            } else {
                $valueString = $value->toSql();
            }
            $valueString = $value->toSql();
        }
        $valueString = $value;
        if(is_array($value)) {
            if($this->comparator == "between") {
                $valueString = $value[0]." AND ".$value[1];
            } else {
                $valueString = implode(",",$value);
            }
        }
        return $prefix.$valueString.$suffix;
    }

    public function prepareValues($value,&$values=[],$column=null) {
        if($value instanceof Value) {
            if($column) {
                $column->typeFix($value);
            }
            if($value->needsPreparing() && $value->values()) {
                $valueValues = $value->values();
                if(is_array($valueValues)) {
                    foreach($valueValues as $val) {
                        if(in_array($this->comparator,['in','nin'])) {
                            $values[] = $val;
                        } else {
                            $this->prepareValues($val,$values);
                        }
                    }
                } else {
                    if(empty($valueValues)) {
                        $values[] = null;
                    } else {
                        $values[] = $this->prefixSuffix($valueValues);
                    }
                    
                }
            }
        } elseif(is_array($value)) {
            foreach($value as $val) {
                $this->prepareValues($val,$values);
            }
        } else {
            if($column) {
                $column->typeFix($value);
            }
            $values[] = $this->prefixSuffix($value);
        }
        return $values;
    }

    public function prepareSql($value,&$values=null,$column=null) {
        if($values !== null) {
            $this->prepareValues($value,$values,$column);
        }
        $prefixString = $suffixString = "";
        if(in_array($this->comparator,['in','nin'])) {
            $prefixString = "(";
            $suffixString = ")";
        }
        if($value instanceof Value) {
            return $prefixString.$value->toSql($this->comparator).$suffixString;
        }
        $string = $value;
        if(is_array($value)) {
            if($this->comparator == "between") {
                $string = $value[0]." AND ".$value[1];
            } else {
                $string = implode(",",$value);
            }
        }
        return $prefixString.$this->prefixSuffix($string).$suffixString;
    }

    private function prefixSuffix($string) {
        if($string === null) {
            return null;
        }
        $prefix = $suffix = "";
        if(isset(self::$prefixSuffix[$this->comparator])) {
            $prefix = self::$prefixSuffix[$this->comparator][0];
            $suffix = self::$prefixSuffix[$this->comparator][1];
        }
        return $prefix.$string.$suffix;
    }
}