<?php
namespace boru\query;

use boru\dhdb\dhDB;
use boru\dhdb\core\Row;
use boru\query\models\Tables;
use boru\query\models\Column;
use boru\query\models\Columns;
use boru\query\models\Value;
use boru\query\models\OrderBy;
use boru\query\models\Condition;
use boru\query\models\Conditions;
use boru\query\models\interfaces\SelectInterface;
use boru\query\dtos\ConditionDTO;

class Query {

    const MODE_SELECT = "SELECT";
    const MODE_UPDATE = "UPDATE";
    const MODE_DELETE = "DELETE";
    const MODE_INSERT = "INSERT";
    const MODE_COUNT = "COUNT";


    public $queryId = null;

    private $db = null;
    private static $staticDB = null;

    protected $pretty = false;
    /** @var Tables */
    protected $tables;
    /** @var Columns */
    protected $columns;
    /** @var array */
    protected $rawColumns = [];
    /** @var Conditions */
    protected $set;
    /** @var Conditions */
    protected $where;
    /** @var SelectInterface[] */
    protected $groupBy = [];
    /** @var Conditions */
    protected $having;
    /** @var OrderBy */
    protected $orderBy;
    protected $limitStart = 0;
    protected $limitCount = 0;

    protected $extraClause = [];
    
    protected $queryVerb = "SELECT";
    protected $fromVerb = "FROM";

    protected $queryMode = "SELECT";

    protected $entityDefinition = null;

    public function __construct($tables = [], $columns = [], $db = null) {
        $this->queryId = uniqid();
        $this->where = new Conditions();
        $this->where->setQuery($this);
        $this->set = new Conditions();
        $this->set->setQuery($this);
        $this->having = new Conditions();
        $this->having->setQuery($this);
        $this->orderBy = new OrderBy();
        $this->orderBy->setQuery($this);
        $this->tables = new Tables($tables);
        $this->tables->setQuery($this);
        $this->columns = new Columns($columns);
        $this->columns->setQuery($this);
    }
    public function __clone() {
        $this->where = clone $this->where;
        $this->where->setQuery($this);
        $this->set = clone $this->set;
        $this->set->setQuery($this);
        $this->having = clone $this->having;
        $this->having->setQuery($this);
        $this->orderBy = clone $this->orderBy;
        $this->orderBy->setQuery($this);
        $this->tables = clone $this->tables;
        $this->tables->setQuery($this);
        $this->columns = clone $this->columns;
        $this->columns->setQuery($this);
    }

    public function db($db=null) {
        if($db !== null) {
            $this->db = $db;
            return $this;
        }
        if($this->db === null) {
            $this->db = static::staticDB();
        }
        return $this->db;
    }

    public function pretty($pretty=true) {
        $this->pretty = $pretty ? true : false;
        return $this;
    }

    public function join($table, $on, $type="INNER JOIN") {
        return $this->tables->join($table, $on, $type);
    }

    public function from(...$tables) {
        foreach($tables as $table) {
            $this->tables->addTable($table);
        }
        return $this;
    }
    public function tables(...$tables) {
        foreach($tables as $table) {
            $this->tables->addTable($table);
        }
        return $this;
    }

    public function columns(...$columns) {
        foreach($columns as $column) {
            $this->columns->addColumn($column);
        }
        return $this;
    }

    public function orderBy($column, $direction="ASC") {
        $this->orderBy->add($column, $direction);
        return $this;
    }

    public function groupBy($column) {
        $this->groupBy[] = $column;
        return $this;
    }

    public function limit(...$args) {
        if(count($args) == 1) {
            $this->limitCount = $args[0];
        } elseif(count($args) == 2) {
            $this->limitStart = $args[0];
            $this->limitCount = $args[1];
        } else {
            throw new \Exception("Invalid number of arguments for limit, expected 1 or 2, got ".count($args));
        }
        return $this;
    }

    public function entityDefinition($definition=null) {
        if($definition !== null) {
            $this->entityDefinition = $definition;
            return $this;
        }
        return $this->entityDefinition;
    }

    public function resetWhere() {
        $this->where = new Conditions();
        return $this;
    }

    public function setWhere($where) {
        if($where instanceof Conditions) {
            $this->where = $where;
        } else {
            throw new \Exception("Invalid where condition, expected Conditions, got ".get_class($where));
        }
        $this->where = $where;
        return $this;
    }
    public function setSet($set) {
        if($set instanceof Conditions) {
            $this->set = $set;
        } else {
            throw new \Exception("Invalid set condition, expected Conditions, got ".get_class($set));
        }
        $this->set = $set;
        return $this;
    }

    /**
     * @param string|Condition $columnName
     * @param string|null $operator
     * @param string|null $value
     * @return Query
     */
    public function oldWhere($columnName, $operator=null, $value=null) {
        if($columnName instanceof Conditions) {
            if($this->where->count() == 0) {
                $this->where = $columnName;
            } else {
                $this->where->addCondition($columnName);
            }
        } elseif($columnName instanceof Condition) {
            $this->where->addCondition($columnName);
        } else {
            if(is_array($columnName)) {
                $dto = self::condtitionToDTO($columnName);
                foreach($dto as $condition) {
                    $this->where->addCondition($condition);
                }
            } else {
                $this->where->addCondition([$columnName,$operator,$value]);
            }
        }
        return $this;
    }

    /**
     * Accepts a variety of formats for conditions:
     * 1. "column","=","value","column2","=","value2",...
     * 2. ["column","=","value"],["column","=","value"]
     * 3. [["column","=","value"],["column","=","value"]]
     * @param string|Condition|array $conditions
     * @return Query
     */
    public function where(...$conditions) {
        if(!is_array($conditions[0]) && count($conditions) % 3 == 0) {
            //if $conditions is in "column","=","value" format, turn it into a ConditionDTO object
            $dto = [];
            for($i=0;$i<count($conditions);$i+=3) {
                $dto[] = new ConditionDTO($conditions[$i],$conditions[$i+1],$conditions[$i+2]);
            }
            $conditions = $dto;
        }elseif(is_array($conditions[0]) && count($conditions[0]) == 3 && !is_array($conditions[0][0])) {
            //if $conditions is in ["column","=","value"] format, turn it into a ConditionDTO object
            $conditions = [new ConditionDTO($conditions[0][0],$conditions[0][1],$conditions[0][2])];
        } elseif(is_array($conditions[0]) && count($conditions[0]) % 3 == 0 && !is_array($conditions[0][0])) {
            //if $conditions is in ["column1","=","value1","column2","=","value2"] format, turn it into a ConditionDTO object
            $dto = [];
            for($i=0;$i<count($conditions[0]);$i+=3) {
                $dto[] = new ConditionDTO($conditions[0][$i],$conditions[0][$i+1],$conditions[0][$i+2]);
            }
            $conditions = $dto;
        }
        foreach($conditions as $condition) {
            if($condition instanceof Conditions) {
                if($this->where->count() == 0) {
                    $this->where = $condition;
                } else {
                    $this->where->addCondition($condition);
                }
            } elseif($condition instanceof Condition) {
                $this->where->addCondition($condition);
            } else {
                if(is_array($condition)) {
                    $dto = self::condtitionToDTO($condition);
                    foreach($dto as $condition) {
                        $this->where->addCondition($condition);
                    }
                } else {
                    $this->where->addCondition($condition);
                }
            }
        }
        return $this;
    }

    public static function condtitionToDTO($conditions) {
        //This function should take any the following 3 structures and turn them into an array of ConditionDTO objects:
        // 1. ["column","=","value"]
        // 2. [["column","=","value"],["column","=","value"]]
        // 3. [[["column","=","value"],["column","=","value"]]]
        $dto = [];
        if(is_array($conditions)) {
            if(count($conditions) == 3 && !is_array($conditions[0])) {
                $dto[] = new ConditionDTO($conditions[0],$conditions[1],$conditions[2]);
            } else {
                foreach($conditions as $condition) {
                    if(is_array($condition)) {
                        if(count($condition) == 3 && !is_array($condition[0])) {
                            $dto[] = new ConditionDTO($condition[0],$condition[1],$condition[2]);
                        } else {
                            foreach($condition as $subCondition) {
                                $dto[] = new ConditionDTO($subCondition[0],$subCondition[1],$subCondition[2]);
                            }
                        }
                    }
                }
            }
        }
        return $dto;
    }

    /**
     * @param string|Condition $columnName
     * @param string|null $value
     * @return Query
     */
    public function set($columnName, $value=null) {
        if($columnName instanceof Conditions) {
            if($this->set->count() == 0) {
                $this->set = $columnName;
            } else {
                $this->set->addCondition($columnName);
            }
        } elseif($columnName instanceof Condition) {
            $this->set->addCondition($columnName);
        } else {
            $this->set->addCondition([$columnName,"=",$value]);
        }
        return $this;
    }

    /**
     * @param string|Condition $columnName
     * @param string|null $operator
     * @param string|null $value
     * @return Query
     */
    public function having(...$conditions) {
        if(is_array($conditions[0]) && count($conditions[0]) == 3 && !is_array($conditions[0][0])) {
            //if $conditions is in ["column","=","value"] format, turn it into a ConditionDTO object
            $conditions = [new ConditionDTO($conditions[0][0],$conditions[0][1],$conditions[0][2])];
        } elseif(is_array($conditions[0]) && count($conditions[0]) % 3 == 0 && !is_array($conditions[0][0])) {
            //if $conditions is in ["column1","=","value1","column2","=","value2"] format, turn it into a ConditionDTO object
            $dto = [];
            for($i=0;$i<count($conditions[0]);$i+=3) {
                $dto[] = new ConditionDTO($conditions[0][$i],$conditions[0][$i+1],$conditions[0][$i+2]);
            }
            $conditions = $dto;
        }
        foreach($conditions as $condition) {
            if($condition instanceof Conditions) {
                if($this->having->count() == 0) {
                    $this->having = $condition;
                } else {
                    $this->having->addCondition($condition);
                }
            } elseif($condition instanceof Condition) {
                $this->having->addCondition($condition);
            } else {
                if(is_array($condition)) {
                    $dto = self::condtitionToDTO($condition);
                    foreach($dto as $condition) {
                        $this->having->addCondition($condition);
                    }
                } else {
                    $this->having->addCondition($condition);
                }
            }
        }
        return $this;
    }

    public function getColumn($columnName) {
        if($columnName === null) {
            return null;
        }
        //check columns first
        $column = $this->columns->getColumn($columnName);
        if($column) {
            return $column;
        }
        return $this->tables->column($columnName);
    }

    /**
     * Sets the query mode. If no argument is provided, returns the current mode
     * @param string $mode Options: SELECT, UPDATE, DELETE, INSERT, COUNT
     * @return string|false
     */
    public function mode($mode=null) {
        if($mode !== null) {
            $this->queryMode = strtoupper($mode);
        }
        return $this->queryMode ? strtoupper($this->queryMode) : false;
    }


    public function toSql(&$values=[]) {
        if($this->queryMode == Query::MODE_SELECT) {
            return $this->toSelectSql($values);
        } elseif($this->queryMode == Query::MODE_COUNT) {
            return $this->toCountSql($values);
        } elseif($this->queryMode == Query::MODE_UPDATE) {
            return $this->toUpdateSql($values);
        } elseif($this->queryMode == Query::MODE_DELETE) {
            return $this->toDeleteSql($values);
        } elseif($this->queryMode == Query::MODE_INSERT) {
            return $this->toInsertSql($values);
        } else {
            throw new \Exception("Invalid query mode: ".$this->queryMode);
        }
    }
    
    /**
     * Creates a new (blank) Record object for this query
     * @return Record
     */
    public function newRecord() {
        return new Record($this);
    }

    public function toSelectSql(&$values=[]) {
        $queryParts = [];
        $queryParts[] = $this->sqlSelect();
        $queryParts[] = $this->sqlTables("FROM");
        $queryParts[] = $this->sqlWhere("WHERE",$values);
        $queryParts[] = $this->sqlGroupBy();
        $queryParts[] = $this->sqlHaving("HAVING",$values);
        $queryParts[] = $this->sqlOrderBy();
        $queryParts[] = $this->sqlLimit();
        return $this->sqlCleaner($queryParts);
    }
    public function toCountSql(&$values=[]) {
        $queryParts = [];
        $queryParts[] = "SELECT COUNT(*) as `count`";
        $queryParts[] = $this->sqlTables("FROM");
        $queryParts[] = $this->sqlWhere("WHERE",$values);
        $queryParts[] = $this->sqlGroupBy();
        $queryParts[] = $this->sqlHaving("HAVING",$values);
        $queryParts[] = $this->sqlOrderBy();
        $queryParts[] = $this->sqlLimit();
        return $this->sqlCleaner($queryParts);
    }
    public function toUpdateSql(&$values=[]) {
        $queryParts = [];
        $queryParts[] = "UPDATE";
        $queryParts[] = $this->sqlTables("");
        $queryParts[] = $this->sqlSet("SET",$values,",");
        $queryParts[] = $this->sqlWhere("WHERE",$values);
        $queryParts[] = $this->sqlLimit();
        return $this->sqlCleaner($queryParts);
    }
    public function toDeleteSql(&$values=[]) {
        $queryParts = [];
        $queryParts[] = "DELETE";
        $queryParts[] = $this->sqlTables("FROM");
        $queryParts[] = $this->sqlWhere("WHERE",$values);
        $queryParts[] = $this->sqlLimit();
        return $this->sqlCleaner($queryParts);
    }
    public function toInsertSql(&$values=[]) {
        $insertParts = $this->set->insertColumns();
        $values = $insertParts["values"];
        $queryParts = [];
        $queryParts[] = "INSERT INTO";
        $queryParts[] = $this->sqlTables();
        $queryParts[] = "(".$insertParts["columns"].")";
        $queryParts[] = "VALUES (".$insertParts["questions"].")";
        return $this->sqlCleaner($queryParts);
    }

    /**
     * Generates a stub for a SELECT query
     * @return array "sql" => string, "questions" => array
     */
    public function toInsertStub() {
        $insertParts = $this->set->insertColumns();
        $values = $insertParts["values"];
        $queryParts = [];
        $queryParts[] = "INSERT INTO";
        $queryParts[] = $this->sqlTables();
        $queryParts[] = "(".$insertParts["columns"].")";
        //$queryParts[] = "VALUES (".$insertParts["questions"].")";
        $sql = $this->sqlCleaner($queryParts);
        $array["sql"] = $sql;
        $array["questions"] = $insertParts["questions"];
        return $array;
    }

    public function sqlVerb() {
        return $this->queryVerb;
    }
    public function sqlSelect($prefix="SELECT") {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        return $prefix.$this->columns->toSqlSelect();
    }
    public function sqlTables($prefix="",&$values=[]) {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        return $prefix.$this->tables->toSql();
    }

    public function sqlWhere($prefix="WHERE",&$values=[]) {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        if(($where = $this->where->toSql($values))) {
            return $prefix.$where;
        }
    }

    public function sqlSet($prefix="SET",&$values=[],$glue=" AND ") {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        if(($set = $this->set->innerSql($values,$glue))) {
            return $prefix.$set;
        }
    }

    public function sqlHaving($prefix="HAVING",&$values=[]) {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        if(($where = $this->having->toSql($values))) {
            return $prefix.$where;
        }
    }

    public function sqlGroupBy($prefix="GROUP BY") {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        $parts = [];
        foreach($this->groupBy as $column) {
            $sql = "";
            if($column instanceof Column || $column instanceof Value) {
                $sql = $column->toSql();
            } else {
                if(($column = $this->getColumn($column))) {
                    $sql = $column->toSql();
                } else {
                    $column = new Value($column,Value::TYPE_FUNCTION);
                    $sql = $column->toSql();
                }
            }
            $parts[] = $sql;
        }
        if(count($parts) == 0) {
            return "";
        }
        if(empty($prefix)) {
            return implode(",",$parts);
        }
        return $prefix.implode(",",$parts);
    }

    public function sqlLimit($prefix="LIMIT") {
        if(!is_numeric($this->limitStart)) {
            $this->limitStart = 0;
        } else {
            $this->limitStart = intval($this->limitStart);
        }
        if(!is_numeric($this->limitCount)) {
            $this->limitCount = 0;
        } else {
            $this->limitCount = intval($this->limitCount);
        }
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        if($this->limitCount == 0) {
            return "";
        }
        $sql = $prefix.$this->limitStart.", ".$this->limitCount;
        return $sql;
    }

    public function sqlExtraClause($prefix="", &$values=[]) {
        if(!empty($prefix)) { $prefix.=" "; } else { $prefix = ""; }
        $parts = [];
        foreach($this->extraClause as $clause) {
            $parts[] = $clause;
        }
        return $prefix.implode(" ",$parts);
    }

    public function sqlOrderBy($prefix="ORDER BY") {
        return $this->orderBy->toSql($prefix);
    }


    public function sqlCleaner($queryParts=[]) {
        foreach($queryParts as $idx=>$part) {
            if(empty($part)) {
                unset($queryParts[$idx]);
            }
        }
        $implode = $this->pretty ? "\n" : " ";
        return implode($implode,$queryParts);
    }


    public function getColumns() {
        return $this->columns;
    }
    public function getTables() {
        return $this->tables;
    }
    public function getWhere() {
        return $this->where;
    }
    public function getSet() {
        return $this->set;
    }
    public function getHaving() {
        return $this->having;
    }
    public function getOrderBy() {
        return $this->orderBy;
    }
    public function getGroupBy() {
        return $this->groupBy;
    }
    public function getLimit() {
        return [$this->limitStart,$this->limitCount];
    }


    /**
     * @param mixed ...$tables
     * @return Query
     */
    public function update(...$tables) {
        $this->queryMode = "UPDATE";
        if(!empty($tables)) {
            $this->tables(...$tables);
        }
        return $this;
    }

    /**
     * @param mixed ...$tables
     * @return Query
     */
    public function delete(...$tables) {
        $this->queryMode = "DELETE";
        if(!empty($tables)) {
            $this->tables(...$tables);
        }
        return $this;
    }

    /**
     * @param mixed ...$tables
     * @return Query
     */
    public function insert(...$tables) {
        $this->queryMode = "INSERT";
        if(!empty($tables)) {
            $this->tables(...$tables);
        }
        return $this;
    }

    /**
     * The columns to select, usually followed by a call to ->from()
     * @param mixed ...$columns
     * @return Query
     */
    public function select(...$columns) {
        $this->queryMode = "SELECT";
        if(!empty($columns)) {
            $this->columns(...$columns);
        }
        return $this;
    }

    /**
     * The columns to count, usually followed by a call to ->from()
     * @param mixed ...$columns
     * @return Query
     */
    public function count(...$columns) {
        $this->queryMode = "COUNT";
        if(!empty($columns)) {
            $this->columns(...$columns);
        }
        return $this;
    }






    //output functions

    public function run($queryMode=null) {
        if($queryMode !== null) {
            $this->queryMode = $queryMode;
        }
        $values = [];
        $sql = $this->toSql($values);
        $db = $this->db();
        return $db->run($sql,$values);
    }

    
    /**
     * Runs the query and returns a single row as an associative array
     * @return array
     */
    public function toCount($queryMode=null) {
        $res = $this->run("COUNT");
        $row = $res->next();
        return $row->asArray()['count'];
    }

    
    /**
     * Runs the query and returns a statement object
     * @return \boru\dhdb\core\Statement
     */
    public function toResult($queryMode=null) {
        return $this->run($queryMode);
    }
    /**
     * Runs the query and returns a statement object
     * @return \boru\dhdb\core\Statement
     */
    public function asResult($queryMode=null) {
        return $this->run($queryMode);
    }

    /**
     * Runs the query and returns a single row as an associative array
     * @return \boru\dhdb\core\Row|\boru\dhdb\core\Row[]
     */
    public function toRows($queryMode=null,$asGenerator=true) {
        if($asGenerator) {
            return $this->asRowGenerator($queryMode);
        }
        $res = $this->run($queryMode);
        $rows = [];
        while($row = $res->next()) {

            $rows[] = $row;
            
        }
        return $rows;
    }
    /**
     * Runs the query and returns a single row as an associative array
     * @return \boru\dhdb\core\Row|\boru\dhdb\core\Row[]
     */
    public function asRows($queryMode=null,$asGenerator=true) {
        return $this->toRows($queryMode,$asGenerator);
    }
    private function asRowGenerator($queryMode=null) {
        $res = $this->run($queryMode);
        while($row = $res->next()) {
            yield $row;
        }
    }

    /**
     * Runs the query and returns a single Record object
     * @return Record
     */
    public function toRecord($queryMode=null) {
        $res = $this->run($queryMode);
        $this->limitCount = 1;
        $rowArray = [];
        while($row = $res->next()) {
            $rowArray = $row->asArray();
        }
        return new Record($this,$rowArray);
    }
     /**
     * Runs the query and returns a single Record object
     * @return Record
     */
    public function asRecord($queryMode=null) {
        return $this->toRecord($queryMode);
    }

    /**
     * Runs the query and returns an array of Record objects
     * @return Record[]
     */
    public function toRecords($queryMode=null,$asGenerator=true) {
        if($asGenerator) {
            return $this->asRecordGenerator($queryMode);
        }
        $res = $this->run($queryMode);
        $records = [];
        while($row = $res->next()) {
            $record = new Record($this,$row->asArray());
            $records[] = $record;
        }
        return $records;
    }
    /**
     * Runs the query and returns an array of Record objects
     * @return Record[]
     */
    public function asRecords($queryMode=null,$asGenerator=true) {
        return $this->toRecords($queryMode,$asGenerator);
    }
    private function asRecordGenerator($queryMode=null) {
        $res = $this->run($queryMode);
        while($row = $res->next()) {
            yield new Record($this,$row->asArray());
        }
    }

    public function toEntities($queryMode=null,$entityName=null,$asGenerator=true) {
        if($asGenerator) {
            return $this->asEntityGenerator($queryMode,$entityName);
        }
        $res = $this->run($queryMode);
        $entities = [];
        $definition = $this->entityDefinition();
        $class = $definition->className();
        while($row = $res->next()) {
            if($entityName) {
                $entity = Factory::instance($entityName,$row->asArray());
            } else {
                $entity = new $class($row->asArray(),$definition);
            }
            $records[] = $entity;
        }
        return $records;
    }
    public function asEntities($queryMode=null,$entityName=null,$asGenerator=true) {
        return $this->toEntities($queryMode,$entityName,$asGenerator);
    }
    private function asEntityGenerator($queryMode=null,$entityName=null) {
        $res = $this->run($queryMode);
        $definition = $this->entityDefinition();
        $class = $definition->className();
        while($row = $res->next()) {
            if($entityName) {
                $entity = Factory::instance($entityName,$row->asArray());
            } else {
                $entity = new $class($row->asArray(),$definition);
            }
            yield $entity;
        }
    }

    public function toEntity($queryMode=null,$entityName=null) {
        $res = $this->run($queryMode);
        $this->limitCount = 1;
        $rowArray = [];
        while($row = $res->next()) {
            $rowArray = $row->asArray();
        }
        $definition = $this->entityDefinition();
        $class = $definition->className();
        if($entityName) {
            return Factory::instance($entityName,$rowArray);
        } else {
            return new $class($rowArray,$definition);
        }
    }
    public function asEntity($queryMode=null,$entityName=null) {
        return $this->toEntity($queryMode,$entityName);
    }


    private function asGenerator($rows) {
        foreach($rows as $row) {
            yield $row;
        }
    }








    //static functions

    public static function staticDB($db=null) {
        if($db !== null) {
            if(is_array($db)) {
                $db = dhDB::instance($db);
            }
            self::$staticDB = $db;
        }
        if(self::$staticDB === null) {
            self::$staticDB = dhDB::instance();
        }
        return self::$staticDB;
    }

    /**
     * @param string $column
     * @param string $direction
     * @return Query
     */
    public static function create() {
        return new static();
    }

    /**
     * Generates a Query object in SELECT mode
     * @param mixed ...$columns The columns to select
     * @return SelectQuery 
     */
    public static function selectQuery(...$columns) {
        $query = new static();
        $query->select(...$columns);
        return $query;
    }

    /**
     * Generates a Query object in UPDATE mode
     * @param mixed ...$tables The tables to update
     * @return SelectQuery 
     */
    public static function updateQuery(...$tables) {
        $query = new static();
        $query->update(...$tables);
        return $query;
    }

    /**
     * Generates a Query object in DELETE mode
     * @param mixed ...$tables The tables to delete from
     * @return SelectQuery 
     */
    public static function deleteQuery(...$tables) {
        $query = new static();
        $query->delete(...$tables);
        return $query;
    }

    /**
     * Generates a Query object in INSERT mode
     * @param mixed ...$tables The tables to insert into
     * @return SelectQuery 
     */
    public static function insertQuery(...$tables) {
        $query = new static();
        $query->insert(...$tables);
        return $query;
    }

    public static function fromQuery($query) {
        $newQuery = new static();
        $newQuery->tables($query->getTables());
        $newQuery->columns($query->getColumns());
        $newQuery->where($query->getWhere());
        $newQuery->orderBy($query->getOrderBy());
        $newQuery->groupBy($query->getGroupBy());
        $newQuery->having($query->getHaving());
        $newQuery->limit($query->getLimit());
        return $newQuery;
    }
}