<?php
namespace boru\query\models;

use boru\dhdb\dhDB;
use boru\query\models\Column;
use boru\query\Query;

class Table {
    /** @var string */
    private $name;
    /** @var string */
    private $alias;
    /** @var string */
    private $idColumn;
    /** @var Column[] */
    private $columns = [];

    /** @var Column[] */
    private static $cache = [];

    public function __construct($name,$alias=null) {
        if(is_array($name) && count($name) == 2) {
            $alias = $name[1];
            $name = $name[0];
        }
        $this->name = $name;
        $columns = self::scanTable($name);
        $this->columns($columns,true);
        if($alias) {
            $this->alias($alias);
        }
    }

    /**
     * 
     * @param array|Column $column 
     * @return this 
     */
    public function addColumn($column,$clone=false) {
        if($clone && $column instanceof Column) {
            $newfield = clone $column;
            $column = $newfield;
        }
        if(is_array($column)) {
            $column = new Column($column);
        }
        $this->columns[$column->name()] = $column;

        //Check if we need to find an ID Field
        if(!$this->idColumn() && ($column->key() == "PRI" || $column->extra() == "auto_increment")) {
            $this->idColumn = $column->name();
            $column->isPrimary(true);
        }

        //Set the table for the field
        $column->table($this);
        return $this;
    }

    private $db;
    public function db($db=null) {
        if($db !== null) {
            $this->db = $db;
        }
        if(is_null($this->db)) {
            $this->db = dhDB::instance();
        }
        return $this->db;
    }

    public function toSqlJoins() {
        if($this->alias) {
            return "`".$this->name."` AS `".$this->alias."`";
        }
        return "`".$this->name."`";
    }
    public function toSql() {
        if($this->alias) {
            return "`".$this->alias."`";
        }
        return "`".$this->name."`";
    }

    /**
     * 
     * @param mixed $name 
     * @return string 
     */
    public function name($name=null) {
        if($name !== null) {
            $this->name = $name;
        }
        return $this->name;
    }

    /**
     * 
     * @param mixed $alias 
     * @return string 
     */
    public function alias($alias=null) {
        if($alias !== null) {
            $this->alias = $alias;
        }
        return $this->alias;
    }

    /**
     * 
     * @return string 
     */
    public function refName() {
        if($this->alias) {
            return $this->alias;
        }
        return $this->name;
    }

    /**
     * 
     * @param mixed $idField 
     * @return string|null
     */
    public function idColumn($idField=null) {
        if($idField !== null) {
            $this->idColumn = $idField;
        }
        return $this->idColumn;
    }

    /**
     * 
     * @param array $fields 
     * @return Field[] 
     */
    public function columns($fields=null,$clone=false) {
        if($fields !== null) {
            if(is_array($fields)) {
                $this->columns = [];
                $this->idColumn = null;
                foreach($fields as $field) {
                    $this->addColumn($field,$clone);
                }
            } else {
                throw new \Exception("Fields must be an array of Field objects");
            }
        }
        return $this->columns;
    }

    /**
     * 
     * @param mixed $columnName 
     * @return Column|null 
     */
    public function column($columnName) {
        if(isset($this->columns[$columnName])) {
            return $this->columns[$columnName];
        }
        return null;
    }

    public function __toString() {
        return $this->toSql();
    }

    private static function scanTable($tableName,$db=null) {
        if(isset(self::$cache[$tableName])) {
            return self::$cache[$tableName];
        }
        if($db === null) {
            $db = Query::staticDB();
        }
        $sql = "SHOW FULL FIELDS FROM `".$tableName."`";
        $sth = $db->run($sql);
        $columns = [];
        while($row = $db->next($sth)) {
            $column = new Column($row->asArray());
            $columns[$column->name()] = $column;
        }
        self::$cache[$tableName] = $columns;
        return $columns;
    }

    private static $instanceCache = [];
    public static function instance($tableName) {
        if(isset(self::$instanceCache[$tableName])) {
            return self::$instanceCache[$tableName];
        }
        $table = new Table($tableName);
        self::$instanceCache[$tableName] = $table;
        return $table;
    }
}