<?php
namespace boru\query;

use boru\query\Query;
use boru\query\models\EntityDefinition;

class Factory {
    /**
     * The key of the array is the entity name
     * @var EntityDefinition[]
     */
    private static $entityNameToDefinition = [];

    /**
     * This method is used to register an entity definition
     * @param EntityDefinition|array|Entity $input 
     * @return EntityDefinition 
     */
    public static function register($input) {
        if(is_array($input)) {
            return self::setEntityDefinition($input);
        } 
        if($input instanceof EntityDefinition) {
            return self::setEntityDefinition($input);
        }
        if($input instanceof Entity) {
            $entityDefinition = EntityDefinition::fromEntity($input);
            return self::setEntityDefinition($entityDefinition);
        }
        throw new \Exception("Invalid entity definition");
    }

    /**
     * This method is used to register a query
     * @param Query $query 
     * @param string $entityName
     * @return EntityDefinition 
     */
    public static function registerQuery($query,$entityName=null,$extraDefinitions=[]) {
        if($entityName === null) {
            $entityName = $query->getTables()->primaryTable()->name();
        }
        $entityDefinition = EntityDefinition::fromQuery($entityName,$query);
        if($extraDefinitions) {
            foreach($extraDefinitions as $key=>$value) {
                $entityDefinition->set($key,$value);
            }
        }
        return self::setEntityDefinition($entityDefinition);
    }

    /**
     * This method is used to get the entity definition
     * @param string $entityName 
     * @return EntityDefinition|false 
     */
    public static function getEntityDefinition($entityName,$throw=false) {
        if(!isset(self::$entityNameToDefinition[$entityName])) {
            if($throw) {
                throw new \Exception("Entity definition not found for ".$entityName);
            }
            return false;
        }
        return self::$entityNameToDefinition[$entityName];
    }

    public static function idField($entityName) {
        $entityDefinition = self::getEntityDefinition($entityName,true);
        return $entityDefinition->idField();
    }
    public static function className($entityName) {
        $entityDefinition = self::getEntityDefinition($entityName,true);
        return $entityDefinition->className();
    }

    /**
     * This method is used to set the entity definition
     * @param EntityDefinition|array $entityDefinition 
     * @return EntityDefinition 
     */
    private static function setEntityDefinition($entityDefinition) {
        
        if(!($entityDefinition instanceof EntityDefinition)) {
            $entityDefinition = new EntityDefinition($entityDefinition);
        }
        if(!$entityDefinition->name()) {
            throw new \Exception("Entity name not set");
        }
        $entityName = $entityDefinition->name();
        if(!isset(self::$entityNameToDefinition[$entityName])) {
            self::$entityNameToDefinition[$entityName] = $entityDefinition;
        }
        return self::$entityNameToDefinition[$entityName];
    }

    public static function entities() {
        return array_keys(self::$entityNameToDefinition);
    }
    public static function instance($entityName,$data=[]) {
        $entityDefinition = self::getEntityDefinition($entityName,true);
        $entityClass = $entityDefinition->className();
        return new $entityClass($data,$entityDefinition);
    }
    public static function instanceById($entityName,$dataOrId=[]) {
        $entityDefinition = self::getEntityDefinition($entityName,true);
        $idColumn = $entityDefinition->idField();
        if(is_array($dataOrId) && isset($dataOrId[$idColumn])) {
            $data = $dataOrId[$idColumn];
        } elseif(is_array($dataOrId)) {
            $data = $dataOrId;
        }else {
            $data = $dataOrId;
        }
        $entityClass = $entityDefinition->className();
        return new $entityClass($data,$entityDefinition);
    }

    public static function create($entityName) {
        $entity = self::instance($entityName);
        return $entity;
    }

    /**
     * This method is used to search for entities
     * @param string $entityName 
     * @param array $limit 
     * @param mixed ...$conditions 
     * @return Entity[]|false 
     */
    public static function search($entityName,$limit=[0,10],...$conditions) {
        self::getEntityDefinition($entityName,true);
        $query = self::query($entityName);
        $query->limit($limit);
        $query->where($conditions);
        $results = [];
        foreach($query->toRows() as $row) {
            $results[] = self::instanceById($entityName,$row->asArray());
        }
        return !empty($results) ? $results : [];
    }

    public static function searchOne($entityName,...$conditions) {
        $idField = self::idField($entityName);
        $query = self::query($entityName);
        $query->limit(1);
        $query->where($conditions);
        $row = $query->toRow();
        if($row) {
            $id = $row->get($idField,false);
            return self::instanceById($entityName,$row->asArray());
        }
        return false;
    }

    /**
     * Returns a select query for the entity
     * @param string $entityName 
     * @return Query 
     */
    public static function query($entityName,$querymode="SELECT") {
        $defintion = self::getEntityDefinition($entityName,true);
        $query = $defintion->query();
        $query->mode($querymode);
        return $query;
    }
}