<?php
namespace boru\query\models;

use boru\query\models\interfaces\SelectInterface;

class Value extends BaseQuery {
    private $value;
    private $type;

    private $operator;

    private $column;

    const TYPE_VALUE = "value";
    const TYPE_COLUMN = "column";
    const TYPE_FUNCTION = "function";
    const TYPE_SUBQUERY = "subquery";

    public function __construct($value,$type=null, $column = null) {
        if($type===null) {
            $type = self::TYPE_VALUE;
        }
        $this->type($type);
        $this->value($value);
        $this->column($column);
    }

    public function type($type=null) {
        if($type!==null) {
            if(!in_array($type,[self::TYPE_VALUE,self::TYPE_FUNCTION,self::TYPE_SUBQUERY,self::TYPE_COLUMN])) {
                throw new \Exception("Invalid value type, ".$type);
            }
            $this->type = $type;
        }
        return $this->type;
    }
    public function value($value=null) {
        if($value!==null) {
            $this->value = $value;
        }
        return $this->value;
    }
    public function column($column = null)
    {
        if ($column !== null) {
            $this->column = $column;
        }
        return $this->column;
    }


    public function needsPreparing() {
        return $this->type() == self::TYPE_VALUE;
    }

    public function toSqlSelect() {
        if($this->type() == self::TYPE_COLUMN) {
            return "`".$this->value."`";
        }
        return $this->sqlString();
    }

    /**
     * This function is used to generate a placeholder string for a prepared statement.
     * @param array|string|null $input If an array, it will be appended with the placeholder string. If a string, it will be concatenated with the placeholder string.
     * @return string 
     */
    public function toSql(&$values = null,$operator=null) {
        if($values === null) {
            $values = [];
        }
        $sql = $this->sqlString($values,$operator);
        return $sql;
    }

    /**
     * This function is used to populate an array with the value of the field for use in a prepared statement.
     * @param array $values Values array to add the value to
     * @return array The values array with the value added
     */
    public function values($values=null) {
        if($this->type() == self::TYPE_SUBQUERY || $this->type() == self::TYPE_FUNCTION || $this->type() == self::TYPE_COLUMN) {
            return $values;
        }
        if($values === null) {
            $values = [];
        }
        if(is_array($this->value)) {
            foreach($this->value as $val) {
                $values[] = $val;
            }
        } else {
            $values[] = $this->value;
        }
        return $values;
    }


    private function sqlString(&$values = [],$operator=null) {
        if($this->type() == self::TYPE_SUBQUERY) {
            return "(".$this->value.")";
        } elseif($this->type() == self::TYPE_FUNCTION) {
            return $this->value;
        } elseif($this->type() == self::TYPE_COLUMN) {
            return "`".$this->value."`";
        }
        if(is_array($this->value)) {
            if($operator == "between") {
                $values[] = $this->value[0];
                $values[] = $this->value[1];
                return "? AND ?";
            } else {
                foreach($this->value as $val) {
                    $values[] = $val;
                }
                return str_repeat("?,",count($this->value)-1)."?";
            }
        } elseif ($this->value === null) {
            // Handle NULL values based on column nullability
            if ($this->column && !$this->column->null()) {
                return "''"; // Return empty string for NOT NULL columns
            } else {
                return "NULL";
            }
        } else {
            if($operator instanceof Operator) {
                $operator->prepareValues($this->value,$values);
            }
            return "?";
        }
    }

    public function __toString() {
        return json_encode([
            "value"=>$this->value,
            "type"=>$this->type
        ]);
    }
    public static function nullValue() {
        return new Value("NULL",Value::TYPE_FUNCTION);
    }
    public static function functionValue($function) {
        return new Value($function,Value::TYPE_FUNCTION);
    }
    public static function subqueryValue($subquery) {
        return new Value($subquery,Value::TYPE_SUBQUERY);
    }
    public static function columnValue($column) {
        return new Value($column,Value::TYPE_COLUMN);
    }
    public static function create($value,$type=null) {
        return new Value($value,$type);
    }
}