<?php
namespace boru\queue\Dhprocess;

use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Entity\QueueItem;
use boru\dhprocess\TaskQueue;

/**
 * DhprocessQueueRunner
 *
 * Uses boru\dhprocess\TaskQueue to process multiple queue items in parallel.
 *
 * Typical usage:
 *
 *   $storage = new MySqlQueueStorage($pdo, 'queue_items');
 *
 *   $runner = new DhprocessQueueRunner(
 *       $storage,
 *       'default',                 // queue name
 *       array(
 *           'numWorkers'   => 5,
 *           'maxQueued'    => 100,
 *           'extendedBar'  => true,
 *           'bootstrapFile'=> __DIR__ . '/worker_bootstrap.php',
 *           'done'         => true,
 *       ),
 *       100                         // batch size per run
 *   );
 *
 *   $processed = $runner->run();
 */
class DhprocessQueueRunner
{
    /** @var QueueStorageInterface */
    protected $storage;

    /** @var string */
    protected $queueName;

    /** @var array */
    protected $dhOptions = array();

    /** @var int */
    protected $batchSize = 100;

    /**
     * @param QueueStorageInterface $storage
     * @param string                $queueName
     * @param array                 $dhOptions  Options for TaskQueue::init()
     * @param int                   $batchSize  Max items to process per run
     */
    public function __construct(QueueStorageInterface $storage, $queueName, array $dhOptions = array(), $batchSize = 0)
    {
        $this->storage   = $storage;
        $this->queueName = $queueName;
        $this->dhOptions = $dhOptions;
        $this->batchSize = (int)$batchSize;
    }

    /**
     * Run one batch of queue items through DHProcess.
     *
     * Returns how many items were actually scheduled & processed.
     *
     * @return int
     */
    public function run()
    {
        if ($this->batchSize < 0) {
            $this->batchSize = 0;
        }

        // Initialize DHProcess TaskQueue once per run
        $options = $this->dhOptions;

        // Provide some safe defaults if not already set
        if (!isset($options['numWorkers'])) {
            $options['numWorkers'] = 5;
        }
        if (!isset($options['maxQueued'])) {
            $options['maxQueued'] = $this->batchSize;
        }
        TaskQueue::init($options);

        // Reserve items from the queue
        $items  = $this->reserveItems($this->batchSize);
        $count  = count($items);

        // Schedule each item as a DHProcess task
        foreach ($items as $item) {
            TaskQueue::task('queue_executor', array(
                $item->getId(),
                $item->getQueueName(),
                $item->getTaskName(),
                $item->getPayload(),
                $item->getAttempts(),
            ))->name('queueItem#' . $item->getId());
        }

        // Wait for all workers to finish
        TaskQueue::wait();

        return $count;
    }

    protected function reserveItems($maxItems=0)
    {
        $items  = array();
        $count  = 0;

        while ($maxItems <= 0 || $count < $maxItems) {
            /** @var QueueItem|null $item */
            $item = $this->storage->reserveNext($this->queueName);

            if (!$item) {
                break; // no more items
            }

            $items[] = $item;
            $count++;
        }

        return $items;
    }
}
