<?php
namespace boru\queue\Dhprocess;

/**
 * DTO for boru\dhprocess TaskQueue options.
 *
 * Wraps the typical keys:
 *   - numWorkers
 *   - maxQueued
 *   - extendedBar
 *   - bootstrapFile
 *   - done
 *
 * and allows arbitrary extra options as well.
 */
class DhOptions
{
    /** @var int */
    protected $numWorkers = 5;

    /** @var int */
    protected $maxQueued = 100;

    /** @var bool */
    protected $extendedBar = true;

    /** @var string|null */
    protected $bootstrapFile;

    /** @var bool */
    protected $done = true;

    /** @var array */
    protected $extra = array();

    /**
     * @param int         $numWorkers
     * @param int         $maxQueued
     * @param bool        $extendedBar
     * @param string|null $bootstrapFile
     * @param bool        $done
     * @param array       $extra
     */
    public function __construct(
        $numWorkers = 5,
        $maxQueued = 100,
        $extendedBar = true,
        $bootstrapFile = null,
        $done = true,
        array $extra = array()
    ) {
        $this->numWorkers    = (int)$numWorkers;
        $this->maxQueued     = (int)$maxQueued;
        $this->extendedBar   = (bool)$extendedBar;
        $this->bootstrapFile = $bootstrapFile;
        $this->done          = (bool)$done;
        $this->extra         = $extra;
    }

    /**
     * Convenience factory.
     *
     * @param string|null $bootstrapFile
     * @return DhOptions
     */
    public static function createDefault($bootstrapFile = null)
    {
        $obj = new self();
        $obj->setBootstrapFile($bootstrapFile);
        return $obj;
    }

    /**
     * Hydrate from plain array (for legacy or config sources).
     *
     * @param array $data
     * @return DhOptions
     */
    public static function fromArray(array $data)
    {
        $numWorkers  = isset($data['numWorkers']) ? $data['numWorkers'] : 5;
        $maxQueued   = isset($data['maxQueued']) ? $data['maxQueued'] : 100;
        $extendedBar = isset($data['extendedBar']) ? $data['extendedBar'] : true;
        $bootstrap   = isset($data['bootstrapFile']) ? $data['bootstrapFile'] : null;
        $done        = isset($data['done']) ? $data['done'] : true;

        // Anything not recognized goes into extra
        $extra = $data;
        unset($extra['numWorkers'], $extra['maxQueued'], $extra['extendedBar'], $extra['bootstrapFile'], $extra['done']);

        return new self($numWorkers, $maxQueued, $extendedBar, $bootstrap, $done, $extra);
    }

    /**
     * Export to the array format expected by TaskQueue::init().
     *
     * @return array
     */
    public function toArray()
    {
        $data = array(
            'numWorkers'  => $this->numWorkers,
            'maxQueued'   => $this->maxQueued,
            'extendedBar' => $this->extendedBar,
            'done'        => $this->done,
        );

        if ($this->bootstrapFile !== null) {
            $data['bootstrapFile'] = $this->bootstrapFile;
        }

        // Merge extras without overwriting explicitly set keys
        foreach ($this->extra as $key => $value) {
            if (!array_key_exists($key, $data)) {
                $data[$key] = $value;
            }
        }

        return $data;
    }

    // ---- Getters / setters ----

    public function getNumWorkers()
    {
        return $this->numWorkers;
    }

    public function setNumWorkers($numWorkers)
    {
        $this->numWorkers = (int)$numWorkers;
        return $this;
    }

    public function getMaxQueued()
    {
        return $this->maxQueued;
    }

    public function setMaxQueued($maxQueued)
    {
        $this->maxQueued = (int)$maxQueued;
        return $this;
    }

    public function getExtendedBar()
    {
        return $this->extendedBar;
    }

    public function setExtendedBar($extendedBar)
    {
        $this->extendedBar = (bool)$extendedBar;
        return $this;
    }

    public function getBootstrapFile()
    {
        return $this->bootstrapFile;
    }

    public function setBootstrapFile($bootstrapFile)
    {
        $this->bootstrapFile = $bootstrapFile;
        return $this;
    }

    public function getDone()
    {
        return $this->done;
    }

    public function setDone($done)
    {
        $this->done = (bool)$done;
        return $this;
    }

    /**
     * @return array
     */
    public function getExtra()
    {
        return $this->extra;
    }

    /**
     * @param array $extra
     * @return $this
     */
    public function setExtra(array $extra)
    {
        $this->extra = $extra;
        return $this;
    }

    /**
     * Add a single extra option.
     *
     * @param string $key
     * @param mixed  $value
     * @return $this
     */
    public function withExtra($key, $value)
    {
        $this->extra[$key] = $value;
        return $this;
    }
}
