<?php
namespace boru\queue\Dhprocess;

use boru\queue\Entity\QueueItem;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;

/**
 * Static worker entrypoint used by boru\dhprocess workers.
 *
 * This class is meant to be configured in the worker bootstrap via:
 *
 *   QueueDhWorker::configure(function () {
 *       // return QueueStorageInterface
 *   }, function () {
 *       // return TaskRegistry with all tasks registered
 *   });
 *
 * And then registered with DHProcess:
 *
 *   \boru\dhprocess\Task::register(
 *       'queue_executor',
 *       ['boru\queue\Dhprocess\QueueDhWorker', 'execute']
 *   );
 */
class QueueDhWorker
{
    /** @var callable|null */
    protected static $storageFactory;

    /** @var callable|null */
    protected static $registryFactory;

    /** @var QueueStorageInterface|null */
    protected static $storage;

    /** @var TaskRegistry|null */
    protected static $registry;

    /**
     * Configure factories to build storage & registry inside the worker.
     *
     * @param callable $storageFactory  returns QueueStorageInterface
     * @param callable $registryFactory returns TaskRegistry
     */
    public static function configure($storageFactory, $registryFactory)
    {
        self::$storageFactory  = $storageFactory;
        self::$registryFactory = $registryFactory;
    }

    /**
     * Lazy build / reuse storage instance.
     *
     * @return QueueStorageInterface
     * @throws \RuntimeException
     */
    protected static function getStorage()
    {
        if (self::$storage instanceof QueueStorageInterface) {
            return self::$storage;
        }

        //maybe the storageFactory was configured directly with an instance
        if (self::$storageFactory instanceof QueueStorageInterface) {
            self::$storage = self::$storageFactory;
            return self::$storage;
        }

        if (!is_callable(self::$storageFactory)) {
            throw new \RuntimeException('QueueDhWorker storageFactory is not configured.');
        }

        $storage = call_user_func(self::$storageFactory);

        if (!$storage instanceof QueueStorageInterface) {
            throw new \RuntimeException('QueueDhWorker storageFactory must return QueueStorageInterface.');
        }

        self::$storage = $storage;

        return self::$storage;
    }

    /**
     * Lazy build / reuse TaskRegistry instance.
     *
     * @return TaskRegistry
     * @throws \RuntimeException
     */
    protected static function getRegistry()
    {
        if (self::$registry instanceof TaskRegistry) {
            return self::$registry;
        }

        //maybe the registryFactory was configured directly with an instance
        if (self::$registryFactory instanceof TaskRegistry) {
            self::$registry = self::$registryFactory;
            return self::$registry;
        }

        if (!is_callable(self::$registryFactory)) {
            throw new \RuntimeException('QueueDhWorker registryFactory is not configured.');
        }

        $registry = call_user_func(self::$registryFactory);

        if (!$registry instanceof TaskRegistry) {
            throw new \RuntimeException('QueueDhWorker registryFactory must return TaskRegistry.');
        }

        self::$registry = $registry;

        return self::$registry;
    }

    /**
     * This is the callable that DHProcess executes in worker processes.
     *
     * Signature must match how the parent schedules the task:
     *
     *   TaskQueue::task('queue_executor', [
     *       $item->getId(),
     *       $item->getQueueName(),
     *       $item->getTaskName(),
     *       $item->getPayload(),
     *       $item->getAttempts(),
     *   ]);
     *
     * @param int         $itemId
     * @param string      $queueName
     * @param string      $taskName
     * @param string|null $payloadJson
     * @param int         $attempts
     * @return array  Small status struct, useful for logging / testing.
     */
    public static function execute($itemId, $queueName, $taskName, $payloadJson = null, $attempts = 0)
    {
        return self::processSingle($itemId, $queueName, $taskName, $payloadJson, $attempts);
    }

    /**
     * NEW: batch entry point.
     *
     * @param array $itemsData Array of item payloads:
     *   [
     *     [
     *       'id'         => 123,
     *       'queue_name' => 'default',
     *       'task_name'  => 'demo_task',
     *       'payload'    => '{"foo":"bar"}',
     *       'attempts'   => 0,
     *     ],
     *     ...
     *   ]
     * @return array Array of per-item status results.
     */
    public static function executeBatch(array $itemsData)
    {
        $results = array();

        foreach ($itemsData as $data) {
            $itemId     = isset($data['id']) ? $data['id'] : null;
            $queueName  = isset($data['queue_name']) ? $data['queue_name'] : 'default';
            $taskName   = isset($data['task_name']) ? $data['task_name'] : null;
            $payload    = isset($data['payload']) ? $data['payload'] : null;
            $attempts   = isset($data['attempts']) ? $data['attempts'] : 0;

            $results[] = self::processSingle($itemId, $queueName, $taskName, $payload, $attempts);
        }

        return $results;
    }

    /**
     * This is the callable that DHProcess executes in worker processes.
     *
     * Signature must match how the parent schedules the task:
     *
     *   TaskQueue::task('queue_executor', [
     *       $item->getId(),
     *       $item->getQueueName(),
     *       $item->getTaskName(),
     *       $item->getPayload(),
     *       $item->getAttempts(),
     *   ]);
     *
     * @param int         $itemId
     * @param string      $queueName
     * @param string      $taskName
     * @param string|null $payloadJson
     * @param int         $attempts
     * @return array  Small status struct, useful for logging / testing.
     */
    public static function processSingle($itemId, $queueName, $taskName, $payloadJson = null, $attempts = 0)
    {
        $storage  = self::getStorage();
        $registry = self::getRegistry();

        // Build a minimal QueueItem; parent already set status=processing & started_at.
        $item = new QueueItem();
        $item->setId($itemId);
        $item->setQueueName($queueName);
        $item->setTaskName($taskName);
        $item->setPayload($payloadJson);
        $item->setAttempts((int)$attempts);
        $item->setStatus(QueueItem::STATUS_PROCESSING);

        $task = $registry->get($taskName);

        if (!$task) {
            $storage->markError($item, json_encode(array(
                'error' => 'Task not found in worker: ' . $taskName,
            )));

            return array(
                'status'  => 'error',
                'reason'  => 'task_not_found',
                'item_id' => $itemId,
            );
        }

        try {
            // AbstractTask::execute() will handle JSON decode / encode.
            $result = $task->execute($item);

            $storage->markDone($item, $result);

            return array(
                'status'  => 'ok',
                'item_id' => $itemId,
            );
        } catch (\Exception $e) {
            // Increment attempts on error
            $item->setAttempts($item->getAttempts() + 1);

            $storage->markError($item, json_encode(array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            )));

            return array(
                'status'    => 'error',
                'item_id'   => $itemId,
                'exception' => $e->getMessage(),
            );
        }
    }
}
