<?php
namespace boru\queue;

use boru\dhdb\dhDB;
use boru\queue\Task\TaskRegistry;
use boru\queue\Dhprocess\DhOptions;
use boru\queue\App\QueueApp;
use boru\queue\Dhprocess\QueueDhWorker;
use boru\queue\Task\TaskInterface;

/**
 * Concrete configurable Queue instance.
 *
 * QueueApp defines the lifecycle.
 * Queue provides the real configuration surface.
 *
 * Everything is optional—if you don't configure something,
 * QueueApp defaults will kick in.
 */
class Queue extends QueueApp
{
    /** @var dhDB|array|null */
    protected $dbConfig;

    /** @var string|null */
    protected $tableName;

    /** @var string */
    protected $queueName = 'default';

    /** @var TaskRegistry|null */
    protected $taskRegistry;

    /** @var DhOptions|array|null */
    protected $dhOptions;

    /** @var int */
    protected $syncSleepSeconds = 2;

    /**
     * ---- CONFIG SETTERS ----
     */

    /**
     * @param dhDB $db
     * @return $this
     */
    public function setDb(dhDB $db)
    {
        // Store dhDB instance directly
        $this->dbConfig = $db;
        return $this;
    }

    /**
     * @param dhDB|array $dbConfig
     * @return $this
     */
    public function setDbConfig($dbConfig)
    {
        $this->dbConfig = $dbConfig;
        return $this;
    }

    /**
     * @param string $tableName
     * @return $this
     */
    public function setTableName($tableName)
    {
        $this->tableName = $tableName;
        return $this;
    }

    /**
     * @param string $queueName
     * @return $this
     */
    public function setQueueName($queueName)
    {
        $this->queueName = $queueName;
        return $this;
    }

    /**
     * @param TaskRegistry $registry
     * @return $this
     */
    public function setTaskRegistry(TaskRegistry $registry)
    {
        $this->taskRegistry = $registry;
        return $this;
    }

    /**
     * @param DhOptions|array $dhOptions
     * @return $this
     */
    public function setDhOptions($dhOptions)
    {
        $this->dhOptions = $dhOptions;
        return $this;
    }

    /**
     * @param int $seconds
     * @return $this
     */
    public function setSyncSleepSeconds($seconds)
    {
        $this->syncSleepSeconds = (int)$seconds;
        return $this;
    }

    /**
     * ---- GETTERS / OVERRIDES REQUIRED BY QueueApp ----
     */

    /**
     * @return array
     * @throws \RuntimeException
     */
    public function getDbConfig()
    {
        if ($this->dbConfig === null) {
            throw new \RuntimeException("Queue::dbConfig was not set.");
        }
        return $this->dbConfig;
    }

    /**
     * @return string
     * @throws \RuntimeException
     */
    public function getTableName()
    {
        if ($this->tableName === null) {
            throw new \RuntimeException("Queue::tableName was not set.");
        }
        return $this->tableName;
    }

    /**
     * @return string
     */
    public function getQueueName()
    {
        return $this->queueName;
    }

    public function getStorage($queueName = null) {
        $queueName = $queueName ? $queueName : $this->getQueueName();
        return $this->getRunner($queueName)->getStorage();
    }

    /**
     * @return TaskRegistry
     */
    public function getTaskRegistryInstance()
    {
        // QueueApp expects registerTasks($registry)
        // We override registerTasks() to inject our registry instance.
        if ($this->taskRegistry instanceof TaskRegistry) {
            return $this->taskRegistry;
        }

        // No registry provided → create an empty one.
        return new TaskRegistry();
    }

    /**
     * @return DhOptions|array
     */
    public function getDhOptions()
    {
        if ($this->dhOptions !== null) {
            return $this->dhOptions;
        }

        // Default fallback
        return DhOptions::createDefault();
    }

    /**
     * @return int
     */
    public function getSyncSleepSeconds()
    {
        return $this->syncSleepSeconds;
    }

    /**
     * QueueApp expects: registerTasks(TaskRegistry $registry).
     *
     * Here we simply copy tasks from the provided TaskRegistry instance.
     *
     * @param TaskRegistry $registry
     * @return void
     */
    public function registerTasks(TaskRegistry $registry)
    {
        if(!$this->taskRegistry) {
            return $this->setTaskRegistry($registry);
        } else {
            foreach ($this->taskRegistry->getAllTasks() as $taskName => $task) {
                $registry->register($task);
            }
        }

        // If no registry was set, QueueApp will end up with an empty registry.
        // You could throw a warning or exception here if desired.
    }

    public function registerTask(TaskInterface $task)
    {
        if(!$this->taskRegistry) {
            $this->taskRegistry = new TaskRegistry();
        }
        $this->taskRegistry->register($task);
    }

    public function configureWorker()
    {
        QueueDhWorker::configure(
            $this->getStorage(),
            $this->getTaskRegistryInstance()
        );
    }
}
