<?php
namespace boru\queue;

use boru\dhdb\dhDB;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;
use boru\queue\Handler\SynchronousTaskHandler;
use boru\queue\QueueWorker;
use boru\queue\Dhprocess\DhprocessQueueRunner;
use boru\queue\Dhprocess\QueueDhWorker;
use boru\dhprocess\Task;
use boru\queue\Dhprocess\DhOptions;
use boru\queue\Storage\MysqlQueueStorage;

/**
 * Unified Queue Runner for sync and dhprocess workflows.
 *
 * Usage:
 *
 *   $runner = new QueueRunner($storage, $registry, 'default');
 *
 *   // Sync:
 *   $runner->runSyncLoop(2);
 *
 *   // Dhprocess:
 *   $processed = $runner->runDhprocess($dhOptions, 100);
 */
class QueueRunner
{
    /** @var QueueStorageInterface */
    protected $storage;

    /** @var TaskRegistry */
    protected $registry;

    /** @var string */
    protected $queueName;


    /**
     * Factory: build QueueRunner from a dhDB config + table.
     *
     * @param array        $dbConfig   Config for dhDB constructor
     * @param string       $tableName  Queue table name
     * @param TaskRegistry $registry   Pre-configured TaskRegistry
     * @param string       $queueName
     *
     * @return QueueRunner
     */
    public static function fromDhDbConfig(array $dbConfig, $tableName, TaskRegistry $registry, $queueName)
    {
        // Assuming dhDB lives in the global namespace
        $db = new dhDB($dbConfig);

        $storage = new MysqlQueueStorage($db, $tableName);

        return new self($storage, $registry, $queueName);
    }

    public static function fromDb(dhDB $db, $tableName, TaskRegistry $registry, $queueName)
    {
        $storage = new MysqlQueueStorage($db, $tableName);

        return new self($storage, $registry, $queueName);
    }

    /**
     * @param QueueStorageInterface $storage
     * @param TaskRegistry          $registry
     * @param string                $queueName
     */
    public function __construct(QueueStorageInterface $storage, TaskRegistry $registry, $queueName)
    {
        $this->storage   = $storage;
        $this->registry  = $registry;
        $this->queueName = $queueName;
    }

    public function getStorage()
    {
        return $this->storage;
    }

    /**
     * Run the queue in synchronous mode using QueueWorker.
     *
     * @param int $sleepSeconds  Sleep between loops (like your current 2 seconds)
     * @return void
     */
    public function runSyncLoop($sleepSeconds = 2)
    {
        $handler = new SynchronousTaskHandler();
        $worker  = new QueueWorker(
            $this->storage,
            $this->registry,
            $handler,
            $this->queueName
        );

        $worker->runLoop($sleepSeconds);
    }

    public function configureWorker()
    {
        QueueDhWorker::configure($this->storage, $this->registry);
    }

    /**
     * Run the queue using the Dhprocess engine (multi-process).
     *
     * @param array|DhOptions $dhOptions  Options for TaskQueue::init()
     * @param int             $batchSize  Max items to process per run
     * @param int             $itemsPerProcess Number of items to process per DHProcess task
     *
     * @return int  Number of items processed in this run
     */
    public function runAsyncLoop($dhOptions, $batchSize = 100, $itemsPerProcess = 1)
    {
        if ($dhOptions instanceof DhOptions) {
            $optionsArray = $dhOptions->toArray();
        } else {
            // assume legacy array
            $optionsArray = DhOptions::fromArray((array)$dhOptions)->toArray();
        }

        // Configure worker
        QueueDhWorker::configure($this->storage, $this->registry);

        Task::register('queue_executor', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'execute',
        ));

        Task::register('queue_executor_batch', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'executeBatch',
        ));

        // Delegate batch processing to DhprocessQueueRunner
        $runner = new DhprocessQueueRunner(
            $this->storage,
            $this->queueName,
            $optionsArray,
            $batchSize,
            $itemsPerProcess
        );
        return $runner->run();
    }
}
