<?php
namespace boru\queue;

use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;
use boru\queue\Handler\TaskHandlerInterface;
use boru\queue\Entity\QueueItem;

class QueueWorker
{
    /** @var QueueStorageInterface */
    protected $storage;

    /** @var TaskRegistry */
    protected $registry;

    /** @var TaskHandlerInterface */
    protected $handler;

    /** @var string */
    protected $queueName;

    /** @var bool */
    protected $stop = false;

    public function __construct(
        QueueStorageInterface $storage,
        TaskRegistry $registry,
        TaskHandlerInterface $handler,
        $queueName
    ) {
        $this->storage   = $storage;
        $this->registry  = $registry;
        $this->handler   = $handler;
        $this->queueName = $queueName;
    }

    public function stop()
    {
        $this->stop = true;
    }

    /**
     * Run a single cycle: fetch one item, process it if available.
     */
    public function runOnce()
    {
        /** @var QueueItem|null $item */
        $item = $this->storage->reserveNext($this->queueName);

        if (!$item) {
            return;
        }

        $task = $this->registry->get($item->getTaskName());

        if (!$task) {
            $this->storage->markError($item, json_encode(array(
                'error' => 'Task not found: ' . $item->getTaskName(),
            )));
            return;
        }

        try {
            $result = $this->handler->handle($task, $item);
            $this->storage->markDone($item, $result);
        } catch (\Exception $e) {
            $this->storage->markError($item, json_encode(array(
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            )));
        }
    }

    /**
     * Simple loop (you can add sleep, backoff, etc).
     */
    public function runLoop($sleepSeconds = 1)
    {
        while (!$this->stop) {
            $this->runOnce();
            if ($sleepSeconds > 0) {
                sleep($sleepSeconds);
            }
        }
    }
}
