<?php
namespace boru\queue;

use boru\dhdb\dhDB;
use boru\queue\Storage\MysqlQueueStorage;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;
use boru\queue\Task\TaskInterface;
use boru\queue\Task\ClosureTask;
use boru\queue\Entity\QueueItem;
use boru\queue\Dhprocess\DhOptions;
use boru\queue\Dhprocess\QueueDhWorker;
use boru\dhprocess\TaskQueue;
use boru\dhprocess\Task;
use boru\queue\Handler\SynchronousTaskHandler;
use boru\queue\QueueWorker;

/**
 * Simple high-level Queue API.
 *
 * Designed to be usable in both parent and worker processes
 * via a shared "all_init.php".
 */
class Queue
{
    /** @var QueueStorageInterface */
    protected $storage;

    /** @var TaskRegistry */
    protected $registry;

    /** @var string */
    protected $queueName = 'default';

    /** @var DhOptions|null */
    protected $dhOptions;

    /** @var string|null */
    protected $bootstrapFile;

    public function __construct()
    {
        $this->registry = new TaskRegistry();
    }

    /**
     * Configure DB-backed storage (MySQL).
     *
     * @param dhDB   $db
     * @param string $tableName
     * @return $this
     */
    public function withDbStorage(dhDB $db, $tableName)
    {
        $this->storage = new MysqlQueueStorage($db, $tableName);
        return $this;
    }

    /**
     * Optionally inject a pre-built storage implementation.
     *
     * @param QueueStorageInterface $storage
     * @return $this
     */
    public function withStorage(QueueStorageInterface $storage)
    {
        $this->storage = $storage;
        return $this;
    }

    /**
     * Set the logical queue name (for multi-queue setups).
     *
     * @param string $queueName
     * @return $this
     */
    public function setQueueName($queueName)
    {
        $this->queueName = $queueName;
        return $this;
    }

    /**
     * For dhprocess async mode: tell the Queue which file
     * to use as bootstrap for worker processes
     * (typically your all_init.php).
     *
     * @param string $bootstrapFile
     * @return $this
     */
    public function withDhprocessBootstrap($bootstrapFile)
    {
        $this->bootstrapFile = $bootstrapFile;
        return $this;
    }

    public function withBootstrap($bootstrapFile)
    {
        return $this->withDhprocessBootstrap($bootstrapFile);
    }

    /**
     * Set full DhOptions config (optional – Queue will build a default if not provided).
     *
     * @param DhOptions $options
     * @return $this
     */
    public function withDhOptions(DhOptions $options)
    {
        $this->dhOptions = $options;
        return $this;
    }

    /**
     * Register a task by name.
     *
     * - If $task is callable → wrap in ClosureTask.
     * - If $task is TaskInterface → register directly.
     *
     * @param string              $name
     * @param callable|TaskInterface $task
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function registerTask($name, $task)
    {
        if ($task instanceof TaskInterface) {
            $this->registry->register($task);
            return $this;
        }

        if (is_callable($task)) {
            $closureTask = new ClosureTask($name, $task);
            $this->registry->register($closureTask);
            return $this;
        }

        throw new \InvalidArgumentException('Task must be callable or implement TaskInterface.');
    }

    public function getQueueName()
    {
        return $this->queueName;
    }

    /**
     * Get underlying storage (for debugging / advanced use).
     *
     * @return QueueStorageInterface
     * @throws \RuntimeException
     */
    public function getStorage()
    {
        if (!$this->storage) {
            throw new \RuntimeException('Queue storage is not configured. Call withDbStorage() first.');
        }
        return $this->storage;
    }

    /**
     * Get TaskRegistry.
     *
     * @return TaskRegistry
     */
    public function getRegistry()
    {
        return $this->registry;
    }

    /**
     * Configure worker-side static executor.
     * Call this in all_init.php so workers are properly wired.
     *
     * Safe to call in both parent and worker.
     *
     * @return void
     */
    public function configureDhWorker()
    {
        QueueDhWorker::configure($this->getStorage(), $this->getRegistry());
    }

    public function configureWorker()
    {
        $this->configureDhWorker();
    }

    /**
     * Enqueue a new job.
     *
     * @param string $taskName
     * @param array  $payload
     * @return QueueItem
     */
    public function enqueue($taskName, array $payload)
    {
        $storage   = $this->getStorage();
        $queueName = $this->queueName;

        $now = date('Y-m-d H:i:s');
        $item = new QueueItem();
        $item->setQueueName($queueName);
        $item->setTaskName($taskName);
        $item->setPayload(json_encode($payload));
        $item->setStatus(QueueItem::STATUS_QUEUED);
        $item->setAttempts(0);
        $item->setCreatedAt(new \DateTime($now));

        $storage->enqueue($queueName, $taskName, $item);

        return $item;
    }

    /**
     * Run a batch of items using dhprocess (multi-process).
     *
     * @param int $batchSize      Max queue items to process this run
     * @param int $itemsPerProcess How many items a single worker call should handle
     * @param array $extraOptions  Extra DhOptions to merge/override
     * @return int Number of items actually scheduled/processed
     */
    public function runAsyncLoop($batchSize, $itemsPerProcess, $extraOptions=[])
    {
        $storage = $this->getStorage();

        // Build DhOptions if not explicitly provided
        $options = $this->buildDhOptions()->toArray();

        // Sane defaults
        if (!isset($options['numWorkers'])) {
            $options['numWorkers'] = 5;
        }
        if (!isset($options['maxQueued'])) {
            $options['maxQueued'] = $batchSize;
        }
        $options = array_merge($options, $extraOptions);

        TaskQueue::init($options);

        // Parent-side: register dhprocess tasks (not needed in workers)
        Task::register('queue_executor', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'execute',
        ));
        Task::register('queue_executor_batch', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'executeBatch',
        ));

        // Ensure worker is configured in THIS process too (e.g., for local testing).
        $this->configureDhWorker();

        // Reserve items
        $items = array();
        $count = 0;

        while ($count < $batchSize) {
            $item = $storage->reserveNext($this->queueName);
            if (!$item) {
                break;
            }
            $items[] = $item;
            $count++;
        }

        if (empty($items)) {
            return 0;
        }

        $itemsPerProcess = (int)$itemsPerProcess > 0 ? (int)$itemsPerProcess : 1;

        if ($itemsPerProcess === 1) {
            // One queue item per worker task
            foreach ($items as $item) {
                TaskQueue::task('queue_executor', array(
                    $item->getId(),
                    $item->getQueueName(),
                    $item->getTaskName(),
                    $item->getPayload(),
                    $item->getAttempts(),
                ))->name('queueItem#' . $item->getId());
            }
        } else {
            // Multiple items per worker task
            $chunks = array_chunk($items, $itemsPerProcess);

            foreach ($chunks as $chunk) {
                $payloadItems = array();
                $nameParts    = array();

                foreach ($chunk as $item) {
                    $payloadItems[] = array(
                        'id'         => $item->getId(),
                        'queue_name' => $item->getQueueName(),
                        'task_name'  => $item->getTaskName(),
                        'payload'    => $item->getPayload(),
                        'attempts'   => $item->getAttempts(),
                    );
                    $nameParts[] = '#' . $item->getId();
                }

                TaskQueue::task('queue_executor_batch', array($payloadItems))
                    ->name('queueBatch(' . implode(',', $nameParts) . ')');
            }
        }

        TaskQueue::wait();

        return $count;
    }

    /**
     * Run the queue in synchronous (single-process) mode.
     *
     * This mirrors your earlier pattern:
     *   $handler = new SynchronousTaskHandler();
     *   $worker  = new QueueWorker($storage, $registry, $handler, 'default');
     *   $worker->runLoop($sleepSeconds);
     *
     * @param int $sleepSeconds  Sleep time between polling loops
     * @return mixed Whatever QueueWorker::runLoop() returns (often void)
     */
    public function runSyncLoop($sleepSeconds = 2)
    {
        $storage  = $this->getStorage();
        $registry = $this->getRegistry();

        $handler = new SynchronousTaskHandler();
        $worker  = new QueueWorker(
            $storage,
            $registry,
            $handler,
            $this->queueName
        );

        return $worker->runLoop($sleepSeconds);
    }


    /**
     * Internal: build DhOptions using bootstrapFile if needed.
     *
     * @return DhOptions
     */
    protected function buildDhOptions()
    {
        if ($this->dhOptions instanceof DhOptions) {
            return $this->dhOptions;
        }

        $bootstrapFile = $this->bootstrapFile ? $this->bootstrapFile : null;

        return DhOptions::createDefault($bootstrapFile);
    }
}
