<?php
namespace boru\queue\Task;

use boru\queue\Entity\QueueItem;

abstract class AbstractTask implements TaskInterface
{
    /** @var string */
    protected $name;

    /**
     * Optional logger.
     * Expected to have a method: log($level, $message, array $context = array()).
     *
     * @var object|null
     */
    protected $logger;

    /**
     * @param string      $name   Unique task name ("send_email", "sync_accounts", etc.)
     * @param object|null $logger Optional logger with log($level, $message, array $context = array()).
     */
    public function __construct($name, $logger = null)
    {
        $this->name   = $name;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Allow setting / swapping logger at runtime.
     *
     * @param object|null $logger
     */
    public function setLogger($logger)
    {
        $this->logger = $logger;
    }

    /**
     * Main entry point used by the worker.
     * Handles logging, JSON decode/encode and delegates to doExecute().
     *
     * @param QueueItem $item
     * @return string|null Result to store in the queue row (usually JSON)
     */
    final public function execute(QueueItem $item)
    {
        $this->log('info', 'Task starting', array(
            'task'   => $this->getName(),
            'itemId' => $item->getId(),
        ));

        $payload = $this->decodePayload($item);

        $resultData = $this->doExecute($item, $payload);

        $resultString = $this->encodeResult($resultData);

        $this->log('info', 'Task completed', array(
            'task'   => $this->getName(),
            'itemId' => $item->getId(),
        ));

        return $resultString;
    }

    /**
     * Concrete tasks implement this:
     *  - $payload is already decoded JSON (array) or ['_raw' => <string>] on decode failure.
     *  - Return:
     *      - array/stdClass → will be json_encode'd
     *      - string|null    → stored as-is
     *
     * @param QueueItem $item
     * @param array     $payload
     * @return mixed Array/stdClass/string/null
     */
    abstract protected function doExecute(QueueItem $item, array $payload);

    /**
     * Decode the JSON payload from the queue item.
     * Returns an array for convenience. On failure:
     *  - logs a warning
     *  - returns array('_raw' => <original payload>)
     *
     * @param QueueItem $item
     * @return array
     */
    protected function decodePayload(QueueItem $item)
    {
        $raw = $item->getPayload();

        if ($raw === null || $raw === '') {
            return array();
        }
        if(is_array($raw)) {
            $data = $raw;
        } else {
            $data = json_decode($raw, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->log('warning', 'Failed to decode payload JSON', array(
                    'task'   => $this->getName(),
                    'itemId' => $item->getId(),
                    'error'  => $this->getJsonErrorMessage(),
                    'raw'    => $raw,
                ));

                return array('_raw' => $raw);
            }
        }

        if (!is_array($data)) {
            // Normalize non-array JSON (string/number/object) into an array wrapper.
            return array('_value' => $data);
        }

        return $data;
    }

    /**
     * Encode result back to a string for storage.
     *
     * - If $data is string|null → returned as-is.
     * - Else → json_encode($data).
     *
     * @param mixed $data
     * @return string|null
     */
    protected function encodeResult($data)
    {
        if ($data === null) {
            return null;
        }

        if (is_string($data)) {
            return $data;
        }

        // Arrays / objects → JSON
        $json = json_encode($data);

        if (json_last_error() !== JSON_ERROR_NONE) {
            // In worst case, log and fall back to a simple string.
            $this->log('warning', 'Failed to encode result JSON', array(
                'task'  => $this->getName(),
                'error' => $this->getJsonErrorMessage(),
            ));

            return 'RESULT_JSON_ERROR';
        }

        return $json;
    }

    /**
     * Helper to build a structured result payload (common pattern).
     *
     * @param string $status  e.g. "ok", "error", "partial"
     * @param array  $data    Arbitrary result data
     * @param array  $meta    Extra meta (timings, counts, etc.)
     * @return array
     */
    protected function buildResult($status, array $data = array(), array $meta = array())
    {
        return array(
            'status' => $status,
            'data'   => $data,
            'meta'   => $meta,
        );
    }

    /**
     * Lightweight logging helper.
     *
     * @param string $level
     * @param string $message
     * @param array  $context
     */
    protected function log($level, $message, array $context = array())
    {
        if ($this->logger && is_callable(array($this->logger, 'log'))) {
            $this->logger->log($level, $message, $context);
        }
    }

    /**
     * PHP 5.6-safe JSON error message.
     *
     * @return string
     */
    protected function getJsonErrorMessage()
    {
        if (function_exists('json_last_error_msg')) {
            return json_last_error_msg();
        }

        return 'JSON error code: ' . json_last_error();
    }
}
