<?php
namespace boru\queue\Dhprocess;

use boru\dhprocess\Task;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Entity\QueueItem;
use boru\dhprocess\TaskQueue;

/**
 * DhprocessQueueRunner
 *
 * Uses boru\dhprocess\TaskQueue to process multiple queue items in parallel.
 *
 * Typical usage:
 *
 *   $storage = new MySqlQueueStorage($pdo, 'queue_items');
 *
 *   $runner = new DhprocessQueueRunner(
 *       $storage,
 *       'default',                 // queue name
 *       array(
 *           'numWorkers'   => 5,
 *           'maxQueued'    => 100,
 *           'extendedBar'  => true,
 *           'bootstrapFile'=> __DIR__ . '/worker_bootstrap.php',
 *           'done'         => true,
 *       ),
 *       100                         // batch size per run
 *   );
 *
 *   $processed = $runner->run();
 */
class DhprocessQueueRunner
{
    /** @var QueueStorageInterface */
    protected $storage;

    /** @var string */
    protected $queueName;

    /** @var array */
    protected $dhOptions = array();

    /** @var int */
    protected $batchSize = 100;

    /** @var int */
    protected $itemsPerProcess = 1; // NEW

    /**
     * @param QueueStorageInterface $storage
     * @param string                $queueName
     * @param array                 $dhOptions  Options for TaskQueue::init()
     * @param int                   $batchSize  Max items to process per run
     * @param int                   $itemsPerProcess Number of items to process per DHProcess task
     */
    public function __construct(QueueStorageInterface $storage, $queueName, array $dhOptions = array(), $batchSize = 0, $itemsPerProcess = 1)
    {
        $this->storage   = $storage;
        $this->queueName = $queueName;
        $this->dhOptions = $dhOptions;
        $this->batchSize = (int)$batchSize;
        $this->itemsPerProcess = (int)$itemsPerProcess;
    }

    /**
     * Optional setter if you want to configure after construction.
     */
    public function setItemsPerProcess($itemsPerProcess)
    {
        $this->itemsPerProcess = (int)$itemsPerProcess > 0 ? (int)$itemsPerProcess : 1;
        return $this;
    }

    /**
     * Run one batch of queue items through DHProcess.
     *
     * Returns how many items were actually scheduled & processed.
     *
     * @return int
     */
    public function run()
    {
        if ($this->batchSize <= 0) {
            $this->batchSize = 100;
        }

        $options = $this->dhOptions;
        if (!isset($options['numWorkers'])) {
            $options['numWorkers'] = 5;
        }
        if (!isset($options['maxQueued'])) {
            $options['maxQueued'] = $this->batchSize;
        }

        TaskQueue::init($options);

        Task::register('queue_executor', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'execute',
        ));

        Task::register('queue_executor_batch', array(
            'boru\queue\Dhprocess\QueueDhWorker',
            'executeBatch',
        ));


        $items = array();
        $count = 0;

        while ($count < $this->batchSize) {
            /** @var QueueItem|null $item */
            $item = $this->storage->reserveNext($this->queueName);
            if (!$item) {
                break;
            }
            $items[] = $item;
            $count++;
        }

        if (empty($items)) {
            return 0;
        }

        // 🔹 NEW: group items into batches
        if ($this->itemsPerProcess <= 1) {
            // Original behavior: one queue task per item
            foreach ($items as $item) {
                TaskQueue::task('queue_executor', array(
                    $item->getId(),
                    $item->getQueueName(),
                    $item->getTaskName(),
                    $item->getPayload(),
                    $item->getAttempts(),
                ))->name('queueItem#' . $item->getId());
            }
        } else {
            // New behavior: multiple items per worker task
            $chunks = array_chunk($items, $this->itemsPerProcess);

            foreach ($chunks as $chunk) {
                $payloadItems = array();
                $nameParts    = array();

                foreach ($chunk as $item) {
                    $payloadItems[] = array(
                        'id'         => $item->getId(),
                        'queue_name' => $item->getQueueName(),
                        'task_name'  => $item->getTaskName(),
                        'payload'    => $item->getPayload(),
                        'attempts'   => $item->getAttempts(),
                    );
                    $nameParts[] = '#' . $item->getId();
                }

                $taskNameLabel = 'queueBatch(' . implode(',', $nameParts) . ')';
                TaskQueue::task('queue_executor_batch', array($payloadItems))
                    ->name($taskNameLabel);
            }
        }

        TaskQueue::wait();

        return $count;
    }

    protected function reserveItems($maxItems=0)
    {
        $items  = array();
        $count  = 0;

        while ($maxItems <= 0 || $count < $maxItems) {
            /** @var QueueItem|null $item */
            $item = $this->storage->reserveNext($this->queueName);

            if (!$item) {
                break; // no more items
            }

            $items[] = $item;
            $count++;
        }

        return $items;
    }
}
