<?php
namespace boru\queue\App;

use boru\queue\Dhprocess\DhOptions;
use boru\queue\QueueRunner;
use boru\queue\Task\TaskRegistry;
use boru\dhdb\dhDB;
use boru\queue\Task\ClosureTask;
use boru\queue\Storage\MysqlQueueStorage;

/**
 * Abstract QueueApp – encapsulates configuration + task registration.
 *
 * Extend this in each project/context and implement:
 *
 *   - getDbConfig()
 *   - getTableName()
 *   - registerTasks(TaskRegistry $registry)
 *
 * Optionally override:
 *
 *   - getQueueName()
 *   - getDhOptions()
 *   - getSyncSleepSeconds()
 */
abstract class QueueApp
{
    /** @var QueueRunner|null */
    protected $runner;

    /**
     * Return the dhDB config array.
     *
     * Example:
     *   return [
     *     'dbtype' => 'mysql',
     *     'dbhost' => 'localhost',
     *     ...
     *   ];
     *
     * @return array|dhDB
     */
    abstract public function getDbConfig();

    /**
     * Return the queue table name.
     *
     * @return string
     */
    abstract public function getTableName();

    /**
     * Register all tasks for this app in the given TaskRegistry.
     *
     * @param TaskRegistry $registry
     * @return void
     */
    abstract public function registerTasks(TaskRegistry $registry);

    /**
     * Queue name; override if you want multiple named queues.
     *
     * @return string
     */
    public function getQueueName()
    {
        return 'default';
    }

    /**
     * DHProcess options; override per project as needed.
     *
     * Default: DhOptions DTO. If you prefer arrays in older code,
     * QueueRunner will still accept them.
     *
     * @return DhOptions|array
     */
    public function getDhOptions()
    {
        // Default: use DTO
        return DhOptions::createDefault(__DIR__ . '/init_worker.php');
    }

    /**
     * Return the TaskRegistry for this app.
     *
     * @return TaskRegistry
     */
    abstract public function getTaskRegistry();

    /**
     * Sleep seconds for sync loop.
     *
     * @return int
     */
    public function getSyncSleepSeconds()
    {
        return 2;
    }

    /**
     * Build and cache the QueueRunner for this app.
     *
     * @param string|null $queueName
     * @return QueueRunner
     */
    public function getRunner($queueName = null)
    {
        if ($this->runner instanceof QueueRunner) {
            return $this->runner;
        }

        $dbConfig  = $this->getDbConfig();
        $tableName = $this->getTableName();
        $queueName = $queueName ? $queueName : $this->getQueueName();

        $registry = $this->getTaskRegistry() ? $this->getTaskRegistry() :new TaskRegistry();
        $this->registerTasks($registry);

        if($dbConfig instanceof dhDB) {
            // Use factory on QueueRunner
            $this->runner = QueueRunner::fromDb(
                $dbConfig,
                $tableName,
                $registry,
                $queueName
            );

            return $this->runner;
        }

        // Use factory on QueueRunner
        $this->runner = QueueRunner::fromDhDbConfig(
            $dbConfig,
            $tableName,
            $registry,
            $queueName,
        );

        return $this->runner;
    }

    /**
     * Convenience: run in sync mode using QueueRunner.
     *
     * @return void
     */
    public function runSyncLoop($queueName = null)
    {
        $this->getRunner($queueName)->runSyncLoop($this->getSyncSleepSeconds());
    }

    /**
     * Convenience: run in dhprocess mode using QueueRunner.
     *
     * @param int $batchSize
     * @return int processed count
     */
    public function runAsyncLoop($queueName = null, $batchSize = 100, $itemsPerProcess = 1)
    {
        $dhOptions = $this->getDhOptions();

        return $this->getRunner($queueName)->runAsyncLoop($dhOptions, $batchSize, $itemsPerProcess);
    }
}
