<?php
namespace boru\queue\Storage;

use boru\queue\Entity\QueueItem;
use boru\dhdb\dhDB;

class MysqlQueueStorage implements QueueStorageInterface
{
    /** @var dhDB */
    protected $db;
    protected $tableName = 'queue_items';

    public function __construct(dhDB $db, $tableName = 'queue_items')
    {
        $this->db = $db;
        $this->tableName = $tableName;
        if($this->db) {
            $this->createTable();
        }
    }

    public function countsByIdentifier($queueName,$identifier)
    {
        $counts = [
            'queued'     => $this->countQueuedItems($queueName,$identifier),
            'processing' => $this->countProcessingItems($queueName,$identifier),
            'error'      => $this->countErrorItems($queueName,$identifier),
            'done'       => $this->countDoneItems($queueName,$identifier),
        ];
        return $counts;
    }
    public function countQueuedItems($queueName,$identifier=null)
    {
        $sql = "select count(*) as cnt from ".$this->tableName." where queue_name = ? and status = ?";
        $params = [
            $queueName,
            QueueItem::STATUS_QUEUED,
        ];
        if($identifier !== null) {
            $sql .= " and identifier = ?";
            $params[] = $identifier;
        }
        $stmt = $this->db->run($sql, $params);
        while($row = $this->db->next($stmt)) {
            return (int)$row->get("cnt",0);
        }
        return 0;
    }
    public function countProcessingItems($queueName,$identifier=null)
    {
        $sql = "select count(*) as cnt from ".$this->tableName." where queue_name = ? and status = ?";
        $params = [
            $queueName,
            QueueItem::STATUS_PROCESSING,
        ];
        if($identifier !== null) {
            $sql .= " and identifier = ?";
            $params[] = $identifier;
        }
        $stmt = $this->db->run($sql, $params);
        while($row = $this->db->next($stmt)) {
            return (int)$row->get("cnt",0);
        }
        return 0;
    }
    public function countErrorItems($queueName,$identifier=null)
    {
        $sql = "select count(*) as cnt from ".$this->tableName." where queue_name = ? and status = ?";
        $params = [
            $queueName,
            QueueItem::STATUS_ERROR,
        ];
        if($identifier !== null) {
            $sql .= " and identifier = ?";
            $params[] = $identifier;
        }
        $stmt = $this->db->run($sql, $params);
        while($row = $this->db->next($stmt)) {
            return (int)$row->get("cnt",0);
        }
        return 0;
    }
    public function countDoneItems($queueName,$identifier=null)
    {
        $sql = "select count(*) as cnt from ".$this->tableName." where queue_name = ? and status = ?";
        $params = [
            $queueName,
            QueueItem::STATUS_DONE,
        ];
        if($identifier !== null) {
            $sql .= " and identifier = ?";
            $params[] = $identifier;
        }
        $stmt = $this->db->run($sql, $params);
        while($row = $this->db->next($stmt)) {
            return (int)$row->get("cnt",0);
        }
        return 0;
    }

    public function enqueue($queueName, $taskName, $payload, $identifier = null)
    {
        if($payload instanceof QueueItem) {
            $identifier = $payload->getIdentifier();
            $payload = $payload->getPayload();
        }
        if(!is_array($payload) && is_string($payload)) {
            $payload = json_decode($payload, true);
        }
        if(is_array($payload)) {
            if($identifier === null && isset($payload['identifier'])) {
                $identifier = $payload['identifier'];
            } elseif($identifier === null && isset($payload['id'])) {
                $identifier = $payload['id'];
            }
            $payload = json_encode($payload, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        }
        $now = date('Y-m-d H:i:s');
        $sql = "insert into ".$this->tableName." (queue_name, `task_name`, `identifier`, `payload`, `status`, `created_at`) 
                values (?, ?, ?, ?, ?, ?)";
        $params = [
            $queueName,
            $taskName,
            $identifier,
            $payload,
            QueueItem::STATUS_QUEUED,
            $now,
        ];
        $result = $this->db->run($sql, $params);
        $sql = "select * from ".$this->tableName." where queue_name = ? and task_name = ? and identifier = ? and created_at = ?";
        $item = new QueueItem();
        $item->setId($this->db->lastInsertId());
        $item->setQueueName($queueName);
        $item->setTaskName($taskName);
        $item->setIdentifier($identifier);
        $item->setPayload($payload);
        $item->setStatus(QueueItem::STATUS_QUEUED);
        $item->setCreatedAt(new \DateTime($now));
        return $item;
    }

    public function reserveNext($queueName=null)
    {
        $pdo = $this->db->pdo();
        // Your classic "select for update" or atomic claim
        // Simplified here; real impl should handle concurrency.
        $pdo->beginTransaction();

        $stmt = $pdo->prepare("
            SELECT * FROM {$this->tableName}
            WHERE queue_name = :queue_name
              AND status = :status
            ORDER BY id ASC
            LIMIT 1
            FOR UPDATE
        ");
        $stmt->execute(array(
            ':queue_name' => $queueName,
            ':status'     => QueueItem::STATUS_QUEUED,
        ));

        $row = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$row) {
            $pdo->commit();
            return null;
        }

        $now = date('Y-m-d H:i:s');

        $update = $pdo->prepare("
            UPDATE {$this->tableName}
            SET status = :status, started_at = :started_at
            WHERE id = :id
        ");
        $update->execute(array(
            ':status'     => QueueItem::STATUS_PROCESSING,
            ':started_at' => $now,
            ':id'         => $row['id'],
        ));

        $pdo->commit();

        $item = $this->hydrateItem($row);
        $item->setStatus(QueueItem::STATUS_PROCESSING);
        $item->setStartedAt(new \DateTime($now));

        return $item;
    }

    public function save(QueueItem $item)
    {
        $sql = "UPDATE ".$this->tableName." set 
            `identifier` = ?,
            `payload` = ?,
            `result` = ?,
            `status` = ?,
            `started_at` = ?,
            `finished_at` = ?,
            `attempts` = ?
        where id = ?";
        $payload = $item->getPayload();
        if(is_array($payload)) {
            $payload = json_encode($payload, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        }
        $params = [
            $item->getIdentifier(),
            $payload,
            $item->getResult(),
            $item->getStatus(),
            $item->getStartedAt() ? $item->getStartedAt()->format('Y-m-d H:i:s') : null,
            $item->getFinishedAt() ? $item->getFinishedAt()->format('Y-m-d H:i:s') : null,
            $item->getAttempts(),
            $item->getId(),
        ];
        $this->db->run($sql, $params);
    }

    public function pruneOldItems($queueName, \DateTime $before)
    {
        $sql = "DELETE FROM {$this->tableName} 
                WHERE queue_name = ? 
                  AND finished_at < ?";
        $params = [
            $queueName,
            $before->format('Y-m-d H:i:s'),
        ];
        $stmt = $this->db->run($sql, $params);
        return $stmt->rowCount();
    }

    public function markDone(QueueItem $item, $result)
    {
        $item->setStatus(QueueItem::STATUS_DONE);
        $item->setResult($result);
        $item->setFinishedAt(new \DateTime());
        $this->save($item);
    }

    public function markError(QueueItem $item, $errorResult)
    {
        $item->setStatus(QueueItem::STATUS_ERROR);
        $item->setResult($errorResult);
        $item->setFinishedAt(new \DateTime());
        $item->setAttempts($item->getAttempts() + 1);
        $this->save($item);
    }
    
    protected function hydrateItem(array $row)
    {
        $item = new QueueItem();
        $item->setId($row['id']);
        $item->setQueueName($row['queue_name']);
        $item->setTaskName($row['task_name']);
        $item->setIdentifier($row['identifier']);
        $item->setPayload(json_decode($row['payload'], true));
        $item->setResult($row['result']);
        $item->setStatus($row['status']);

        if (!empty($row['created_at'])) {
            $item->setCreatedAt(new \DateTime($row['created_at']));
        }
        if (!empty($row['started_at'])) {
            $item->setStartedAt(new \DateTime($row['started_at']));
        }
        if (!empty($row['finished_at'])) {
            $item->setFinishedAt(new \DateTime($row['finished_at']));
        }
        if (isset($row['attempts'])) {
            $item->setAttempts($row['attempts']);
        }

        return $item;
    }

    protected function formatDate(\DateTime $date = null)
    {
        return $date ? $date->format('Y-m-d H:i:s') : null;
    }

    public function createTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tableName}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `queue_name` varchar(255) NOT NULL,
            `task_name` varchar(255) NOT NULL,
            `identifier` varchar(255) DEFAULT NULL,
            `payload` text,
            `result` text,
            `status` tinyint(1) NOT NULL DEFAULT '0',
            `created_at` datetime DEFAULT NULL,
            `started_at` datetime DEFAULT NULL,
            `finished_at` datetime DEFAULT NULL,
            `attempts` int(11) NOT NULL DEFAULT '0',
            PRIMARY KEY (`id`),
            KEY `idx_queue_status` (`queue_name`,`status`),
            KEY `idx_created_at` (`created_at`),
            KEY `idx_identifier` (`identifier`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
        $this->db->run($sql);
    }
}