<?php

namespace boru\tty;

class Layout
{
    /** @var Terminal */
    protected $tty;

    /** @var StatusLine */
    protected $statusLine;

    /** @var ScrollArea */
    protected $scrollArea;

    /** @var int */
    protected $promptRow;

    /** @var int */
    protected $scrollTop;

    /** @var int */
    protected $scrollHeight;

    private function __construct(
        Terminal $tty,
        StatusLine $statusLine,
        ScrollArea $scrollArea,
        $promptRow,
        $scrollTop,
        $scrollHeight
    ) {
        $this->tty         = $tty;
        $this->statusLine  = $statusLine;
        $this->scrollArea  = $scrollArea;
        $this->promptRow   = (int) $promptRow;
        $this->scrollTop   = (int) $scrollTop;
        $this->scrollHeight = (int) $scrollHeight;
    }

    /**
     * Factory for the pattern:
     * - bottom row: status line
     * - N rows above bottom: scroll area
     * - 1 row above scroll area: prompt/stdout row
     *
     * If the terminal is too small, rows are clamped to >= 1.
     *
     * @param Terminal $tty
     * @param int      $scrollHeight number of rows for scroll area
     * @return Layout
     */
    public static function withStatusAndScroll(Terminal $tty, $scrollHeight)
    {
        $scrollHeight = (int) $scrollHeight;
        if ($scrollHeight < 1) {
            $scrollHeight = 1;
        }

        list($cols, $rows) = $tty->getSize();

        // Bottom row is reserved for the status line
        $statusLine = StatusLine::bottom($tty);

        // Scroll area sits above the status line.
        // Last available row for scroll is rows-1.
        $lastScrollRow = max(1, $rows - 1);
        $scrollTop = max(1, $lastScrollRow - $scrollHeight + 1);

        // Prompt row is one line above the scroll area.
        $promptRow = max(1, $scrollTop - 1);

        $scrollArea = new ScrollArea($tty, $scrollTop, $scrollHeight);

        return new self($tty, $statusLine, $scrollArea, $promptRow, $scrollTop, $scrollHeight);
    }

    // ---------------------------------------------------------------------
    // Accessors
    // ---------------------------------------------------------------------

    /**
     * Move the cursor to a clean line just after the scroll area,
     * clear that line, and leave the cursor there.
     *
     * Safe to call multiple times.
     */
    public function moveToCleanExitLine()
    {
        $scrollTop    = $this->scrollTop;
        $scrollHeight = $this->scrollHeight;
        $lastScroll   = $scrollTop + $scrollHeight - 1;

        $exitRow = $lastScroll + 1;

        list($cols, $rows) = $this->tty->getSize();
        if ($exitRow > $rows) {
            $exitRow = $rows;
        }

        $this->tty->moveCursorTo($exitRow, 1);
        $this->tty->clearLine();
        $this->tty->showCursor();
    }

    /**
     * @return StatusLine
     */
    public function getStatusLine()
    {
        return $this->statusLine;
    }

    /**
     * @return ScrollArea
     */
    public function getScrollArea()
    {
        return $this->scrollArea;
    }

    /**
     * Row index (1-based) used for prompts and "normal stdout".
     *
     * @return int
     */
    public function getPromptRow()
    {
        return $this->promptRow;
    }

    /**
     * Top row (1-based) of the scroll area.
     *
     * @return int
     */
    public function getScrollTop()
    {
        return $this->scrollTop;
    }

    /**
     * Height of the scroll area in rows.
     *
     * @return int
     */
    public function getScrollHeight()
    {
        return $this->scrollHeight;
    }

    // ---------------------------------------------------------------------
    // Convenience output helpers (for prompt/stdout row)
    // ---------------------------------------------------------------------

    public function write($string)
    {
        $this->moveToPromptRow();
        $this->tty->write($string);
    }

    public function writeln($string = '')
    {
        $this->moveToPromptRow();
        $this->tty->write($string . PHP_EOL);
    }

    public function clearPromptRow()
    {
        $this->tty->saveCursor();
        $this->tty->moveCursorTo($this->promptRow, 1);
        $this->tty->clearLine();
        $this->tty->restoreCursor();
    }

    /**
     * Move cursor to prompt row, column 1.
     */
    public function moveToPromptRow()
    {
        $this->tty->moveCursorTo($this->promptRow, 1);
    }

    // ---------------------------------------------------------------------
    // Prompt integration helpers (optional sugar)
    // ---------------------------------------------------------------------

    /**
     * Ask a question on the prompt row using a Prompt helper.
     *
     * @param Prompt     $prompt
     * @param string     $question
     * @param string|null $default
     * @return string|null
     */
    public function ask(Prompt $prompt, $question, $default = null)
    {
        $this->clearPromptRow();
        $this->moveToPromptRow();

        return $prompt->ask($question, $default);
    }

    /**
     * Ask for secret input on the prompt row using a Prompt helper.
     *
     * @param Prompt $prompt
     * @param string $question
     * @return string
     */
    public function secret(Prompt $prompt, $question)
    {
        $this->clearPromptRow();
        $this->moveToPromptRow();

        return $prompt->secret($question);
    }
}
