<?php

namespace boru\tty;

class Prompt
{
    /** @var Terminal */
    protected $tty;

    public function __construct(Terminal $tty)
    {
        $this->tty = $tty;
    }

    /**
     * Ask for a line of input.
     *
     * @param string      $question
     * @param string|null $default
     * @return string|null
     */
    public function ask($question, $default = null)
    {
        $prompt = $question;
        if ($default !== null) {
            $prompt .= ' [' . $default . ']';
        }
        $prompt .= ': ';

        $this->tty->write($prompt);
        $line = $this->readLine();

        if ($line === '' && $default !== null) {
            return $default;
        }

        return $line;
    }

    /**
     * Ask for confirmation (y/n).
     *
     * @param string   $question
     * @param bool|nil $default true, false, or null (no default)
     * @return bool
     */
    public function confirm($question, $default = null)
    {
        $suffix = '';
        if ($default === true) {
            $suffix = ' [Y/n]';
        } elseif ($default === false) {
            $suffix = ' [y/N]';
        } else {
            $suffix = ' [y/n]';
        }

        while (true) {
            $this->tty->write($question . $suffix . ': ');
            $line = strtolower(trim($this->readLine()));

            if ($line === '' && $default !== null) {
                return (bool) $default;
            }

            if ($line === 'y' || $line === 'yes') {
                return true;
            }
            if ($line === 'n' || $line === 'no') {
                return false;
            }

            $this->tty->writeln('Please answer y or n.');
        }
    }

    /**
     * Read a secret value (no echo).
     *
     * @param string $question
     * @return string
     */
    public function secret($question)
    {
        $this->tty->write($question . ': ');

        $this->enableNoEcho();
        $value = $this->readLineRaw();
        $this->disableNoEcho();

        $this->tty->writeln(); // move to next line after password input

        return $value;
    }

    /**
     * Present a simple numbered choice list.
     *
     * @param string $question
     * @param array  $options list of labels
     * @param int|null $defaultIndex 0-based index or null
     * @return string selected option value
     */
    public function choose($question, array $options, $defaultIndex = null)
    {
        if (!$options) {
            throw new \InvalidArgumentException('Options array must not be empty.');
        }

        $this->tty->writeln($question);

        foreach ($options as $idx => $label) {
            $num = $idx + 1;
            $this->tty->writeln('  ' . $num . ') ' . $label);
        }

        $defaultDisplay = '';
        if ($defaultIndex !== null && isset($options[$defaultIndex])) {
            $defaultDisplay = ' [' . ($defaultIndex + 1) . ']';
        }

        while (true) {
            $this->tty->write('Select option' . $defaultDisplay . ': ');
            $line = trim($this->readLine());

            if ($line === '' && $defaultIndex !== null && isset($options[$defaultIndex])) {
                return $options[$defaultIndex];
            }

            if (ctype_digit($line)) {
                $n = (int) $line;
                $idx = $n - 1;
                if (isset($options[$idx])) {
                    return $options[$idx];
                }
            }

            $this->tty->writeln('Invalid selection, try again.');
        }
    }

    // ---------------------------------------------------------------------
    // Internal helpers
    // ---------------------------------------------------------------------

    protected function readLine()
    {
        $fh = $this->getInputStream();
        $line = fgets($fh);
        if ($line === false) {
            return '';
        }
        return rtrim($line, "\r\n");
    }

    protected function readLineRaw()
    {
        // For now, use fgets as well; we could later do char-by-char in raw mode.
        return $this->readLine();
    }

    protected function getInputStream()
    {
        // Use reflection to access Terminal::$in without exposing public API yet
        $ref = new \ReflectionClass($this->tty);
        if ($ref->hasProperty('in')) {
            $prop = $ref->getProperty('in');
            $prop->setAccessible(true);
            return $prop->getValue($this->tty);
        }

        // Fallback
        return defined('STDIN') ? STDIN : fopen('php://stdin', 'r');
    }

    protected function enableNoEcho()
    {
        // On Unix we can use stty; we reuse Terminal::enableRawMode but preserving behavior:
        // to avoid breaking any user raw-mode usage, only switch if not already raw.
        if (stripos(PHP_OS, 'WIN') === 0) {
            // Windows: best-effort, can't easily disable echo without extensions.
            return;
        }

        if (method_exists($this->tty, 'enableRawMode') && !$this->tty->inRawMode()) {
            $this->tty->enableRawMode();
        }
    }

    protected function disableNoEcho()
    {
        if (stripos(PHP_OS, 'WIN') === 0) {
            return;
        }

        if (method_exists($this->tty, 'disableRawMode') && $this->tty->inRawMode()) {
            $this->tty->disableRawMode();
        }
    }
}
