<?php

namespace boru\tty;

class ScrollArea
{
    /** @var Terminal */
    protected $tty;

    /** @var int */
    protected $top;

    /** @var int */
    protected $height;

    /** @var array */
    protected $buffer = array();

    public function __construct(Terminal $tty, $topRow, $height)
    {
        $this->tty    = $tty;
        $this->top    = (int) $topRow;
        $this->height = (int) $height;
    }

    public static function fromBottom(Terminal $tty, $height)
    {
        list($cols, $rows) = $tty->getSize();
        $top = max(1, $rows - (int) $height + 1);
        return new self($tty, $top, $height);
    }

    /**
     * Append a line to the scroll area.
     *
     * @param string $text
     */
    public function addLine($text)
    {
        $this->buffer[] = (string) $text;
        $this->trimBuffer();
        $this->render();
    }

    /**
     * Replace the entire buffer content.
     *
     * @param array $lines
     */
    public function setLines(array $lines)
    {
        $this->buffer = array_values($lines);
        $this->trimBuffer();
        $this->render();
    }

    /**
     * Force re-render of the current buffer.
     */
    public function refresh()
    {
        $this->render();
    }

    /**
     * Clear both the buffer and the on-screen region.
     */
    public function clear()
    {
        $this->buffer = array();

        $this->tty->saveCursor();

        for ($i = 0; $i < $this->height; $i++) {
            $row = $this->top + $i;
            $this->tty->moveCursorTo($row, 1);
            $this->tty->clearLine();
        }

        $this->tty->restoreCursor();
    }

    // ---------------------------------------------------------------------
    // Internal
    // ---------------------------------------------------------------------

    protected function trimBuffer()
    {
        // Keep at most height lines in buffer; consider later keeping more history.
        $max = $this->height;
        $count = count($this->buffer);
        if ($count > $max) {
            $this->buffer = array_slice($this->buffer, $count - $max);
        }
    }

    protected function render()
    {
        list($cols,) = $this->tty->getSize();

        $visible = $this->buffer;
        $visibleCount = count($visible);

        // If fewer than height lines, pad with blanks for clearing.
        if ($visibleCount < $this->height) {
            $pad = $this->height - $visibleCount;
            for ($i = 0; $i < $pad; $i++) {
                $visible[] = '';
            }
        } elseif ($visibleCount > $this->height) {
            $visible = array_slice($visible, $visibleCount - $this->height);
        }

        $this->tty->saveCursor();

        foreach ($visible as $i => $line) {
            $row = $this->top + $i;

            $this->tty->moveCursorTo($row, 1);
            $this->tty->clearLine();

            $text = (string) $line;
            if (strlen($text) > $cols) {
                $text = substr($text, 0, $cols);
            }
            $this->tty->write($text);
        }

        $this->tty->restoreCursor();
    }
}
