<?php

namespace boru\tty;

use boru\tty\Util\Ansi;

class Terminal
{
    /** @var resource */
    protected $in;
    /** @var resource */
    protected $out;

    /** @var bool */
    protected $raw = false;

    /** @var string|null */
    protected $sttyState = null;

    public function __construct($input = null, $output = null)
    {
        if ($input === null) {
            $input = defined('STDIN') ? STDIN : fopen('php://stdin', 'r');
        }
        if ($output === null) {
            $output = defined('STDOUT') ? STDOUT : fopen('php://stdout', 'w');
        }

        $this->in  = $input;
        $this->out = $output;
    }

    // ---------------------------------------------------------------------
    // Capabilities
    // ---------------------------------------------------------------------

    public function isInteractive()
    {
        // stream_isatty exists in newer PHP, fallback to posix_isatty or heuristic
        if (function_exists('stream_isatty')) {
            return @stream_isatty($this->in);
        }

        if (function_exists('posix_isatty')) {
            $meta = stream_get_meta_data($this->in);
            if (!empty($meta['uri'])) {
                return @posix_isatty($this->in);
            }
        }

        // Heuristic: assume interactive if stdin is not a file and not a pipe
        $stat = fstat($this->in);
        if ($stat && isset($stat['mode'])) {
            $mode = $stat['mode'] & 0170000;
            // 0010000 = FIFO, 0100000 = regular file
            if ($mode === 0010000 || $mode === 0100000) {
                return false;
            }
        }

        return true;
    }

    public function supportsColors()
    {
        if (!$this->isInteractive()) {
            return false;
        }

        $env = getenv('TERM');
        if ($env === 'dumb' || $env === '') {
            return false;
        }

        return true;
    }

    /**
     * @return array [int $cols, int $rows]
     */
    public function getSize()
    {
        // Try stty first (most reliable on Unix)
        if ($this->isInteractive()) {
            $stty = $this->runStty('size');
            if ($stty) {
                $parts = preg_split('/\s+/', trim($stty));
                if (count($parts) === 2) {
                    $rows = (int) $parts[0];
                    $cols = (int) $parts[1];
                    if ($rows > 0 && $cols > 0) {
                        return array($cols, $rows);
                    }
                }
            }
        }

        // Fallback to env vars
        $cols = (int) getenv('COLUMNS');
        $rows = (int) getenv('LINES');

        if ($cols > 0 && $rows > 0) {
            return array($cols, $rows);
        }

        // Sensible default
        return array(80, 24);
    }

    // ---------------------------------------------------------------------
    // Modes (raw / cooked)
    // ---------------------------------------------------------------------

    public function enableRawMode()
    {
        if ($this->raw) {
            return;
        }

        if (!$this->isInteractive()) {
            return;
        }

        $current = $this->runStty('-g');
        if ($current === null) {
            return;
        }

        $this->sttyState = trim($current);

        // -echo: do not echo chars
        // -icanon: non-canonical (immediate) input
        $this->runStty('-echo -icanon');

        $this->raw = true;
    }

    public function disableRawMode()
    {
        if (!$this->raw) {
            return;
        }

        if ($this->sttyState !== null) {
            $this->runStty($this->sttyState, true);
        }

        $this->raw      = false;
        $this->sttyState = null;
    }

    public function inRawMode()
    {
        return $this->raw;
    }

    // ---------------------------------------------------------------------
    // Output helpers
    // ---------------------------------------------------------------------

    public function write($string)
    {
        fwrite($this->out, $string);
    }

    public function writeln($string = '')
    {
        fwrite($this->out, $string . PHP_EOL);
    }

    public function flush()
    {
        fflush($this->out);
    }

    // ---------------------------------------------------------------------
    // Cursor & screen
    // ---------------------------------------------------------------------

    public function moveCursorTo($row, $col)
    {
        $this->write(Ansi::cursorPosition((int) $row, (int) $col));
    }

    public function moveCursorBy($dRow, $dCol)
    {
        $this->write(Ansi::cursorMove((int) $dRow, (int) $dCol));
    }

    public function saveCursor()
    {
        $this->write("\x1b7");
    }

    public function restoreCursor()
    {
        $this->write("\x1b8");
    }

    public function clearScreen()
    {
        $this->write(Ansi::clearScreen());
    }

    /**
     * $mode: 0 = clear from cursor right, 1 = clear from cursor left, 2 = entire line
     */
    public function clearLine($mode = 2)
    {
        $this->write(Ansi::clearLine($mode));
    }

    public function hideCursor()
    {
        $this->write(Ansi::hideCursor());
    }

    public function showCursor()
    {
        $this->write(Ansi::showCursor());
    }

    // ---------------------------------------------------------------------
    // Simple styling (ANSI)
    // ---------------------------------------------------------------------

    public function color($fg = null, $bg = null, array $options = array())
    {
        if (!$this->supportsColors()) {
            return;
        }

        $codes = array();

        if ($fg !== null) {
            $codes[] = $this->mapColor($fg, false);
        }

        if ($bg !== null) {
            $codes[] = $this->mapColor($bg, true);
        }

        foreach ($options as $opt) {
            $code = $this->mapOption($opt);
            if ($code !== null) {
                $codes[] = $code;
            }
        }

        $codes = array_filter($codes, 'strlen');
        if ($codes) {
            $this->write(Ansi::sgr($codes));
        }
    }

    public function resetStyle()
    {
        if (!$this->supportsColors()) {
            return;
        }

        $this->write(Ansi::reset());
    }

    // ---------------------------------------------------------------------
    // Internal helpers
    // ---------------------------------------------------------------------

    protected function runStty($options, $restore = false)
    {
        // Only works on Unix-like; skip on Windows
        if (stripos(PHP_OS, 'WIN') === 0) {
            return null;
        }

        $descriptorSpec = array(
            0 => $restore ? $this->in : $this->in,      // stdin
            1 => array('pipe', 'w'),                    // stdout
            2 => array('pipe', 'w'),                    // stderr
        );

        $cmd = 'stty ' . $options;

        $proc = @proc_open($cmd, $descriptorSpec, $pipes);
        if (!is_resource($proc)) {
            return null;
        }

        $out = stream_get_contents($pipes[1]);
        $err = stream_get_contents($pipes[2]);

        foreach ($pipes as $pipe) {
            fclose($pipe);
        }

        $status = proc_close($proc);

        if ($status !== 0 && !$restore) {
            // For non-restore operations, treat failure as null
            return null;
        }

        return $out !== '' ? $out : ($err !== '' ? $err : '');
    }

    protected function mapColor($name, $background)
    {
        $fgMap = array(
            'black'   => 30,
            'red'     => 31,
            'green'   => 32,
            'yellow'  => 33,
            'blue'    => 34,
            'magenta' => 35,
            'cyan'    => 36,
            'white'   => 37,
            'default' => 39,
        );

        $bgMap = array(
            'black'   => 40,
            'red'     => 41,
            'green'   => 42,
            'yellow'  => 43,
            'blue'    => 44,
            'magenta' => 45,
            'cyan'    => 46,
            'white'   => 47,
            'default' => 49,
        );

        $map = $background ? $bgMap : $fgMap;

        return isset($map[$name]) ? $map[$name] : null;
    }

    protected function mapOption($name)
    {
        $map = array(
            'bold'      => 1,
            'dim'       => 2,
            'underline' => 4,
            'blink'     => 5,
            'reverse'   => 7,
        );

        return isset($map[$name]) ? $map[$name] : null;
    }
}
